import { supabase } from './supabase'
import type { RealtimeChannel } from '@supabase/supabase-js'

export class RealtimeManager {
  private channels: Map<string, RealtimeChannel> = new Map()

  // Subscribe to user analytics updates
  subscribeToUserAnalytics(userId: string, callback: (payload: any) => void): RealtimeChannel {
    const channelName = `user-analytics-${userId}`
    
    if (this.channels.has(channelName)) {
      return this.channels.get(channelName)!
    }

    const channel = supabase
      .channel(channelName)
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'analytics',
          filter: `user_id=eq.${userId}`
        },
        callback
      )
      .subscribe()

    this.channels.set(channelName, channel)
    return channel
  }

  // Subscribe to server updates
  subscribeToServerUpdates(serverId: string, callback: (payload: any) => void): RealtimeChannel {
    const channelName = `server-updates-${serverId}`
    
    if (this.channels.has(channelName)) {
      return this.channels.get(channelName)!
    }

    const channel = supabase
      .channel(channelName)
      .on(
        'postgres_changes',
        {
          event: 'UPDATE',
          schema: 'public',
          table: 'servers',
          filter: `id=eq.${serverId}`
        },
        callback
      )
      .subscribe()

    this.channels.set(channelName, channel)
    return channel
  }

  // Subscribe to global analytics (for admin dashboard)
  subscribeToGlobalAnalytics(callback: (payload: any) => void): RealtimeChannel {
    const channelName = 'global-analytics'
    
    if (this.channels.has(channelName)) {
      return this.channels.get(channelName)!
    }

    const channel = supabase
      .channel(channelName)
      .on(
        'postgres_changes',
        {
          event: 'INSERT',
          schema: 'public',
          table: 'analytics'
        },
        callback
      )
      .subscribe()

    this.channels.set(channelName, channel)
    return channel
  }

  // Subscribe to user favorites
  subscribeToUserFavorites(userId: string, callback: (payload: any) => void): RealtimeChannel {
    const channelName = `user-favorites-${userId}`
    
    if (this.channels.has(channelName)) {
      return this.channels.get(channelName)!
    }

    const channel = supabase
      .channel(channelName)
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'favorites',
          filter: `user_id=eq.${userId}`
        },
        callback
      )
      .subscribe()

    this.channels.set(channelName, channel)
    return channel
  }

  // Unsubscribe from a specific channel
  unsubscribe(channelName: string): void {
    const channel = this.channels.get(channelName)
    if (channel) {
      supabase.removeChannel(channel)
      this.channels.delete(channelName)
    }
  }

  // Unsubscribe from all channels
  unsubscribeAll(): void {
    this.channels.forEach((channel, channelName) => {
      supabase.removeChannel(channel)
    })
    this.channels.clear()
  }

  // Get active channels count
  getActiveChannelsCount(): number {
    return this.channels.size
  }

  // Get channel names
  getChannelNames(): string[] {
    return Array.from(this.channels.keys())
  }
}

// Export singleton instance
export const realtimeManager = new RealtimeManager()
