"use client"

import React, { createContext, useContext, useEffect, useState } from 'react'
import { User, Session } from '@supabase/supabase-js'
import { AuthAPI, type DbUser } from './database'

interface AuthContextType {
  user: User | null
  dbUser: DbUser | null
  session: Session | null
  loading: boolean
  signIn: (email: string, password: string) => Promise<{ error: any }>
  signUp: (email: string, password: string, userData: { username: string }) => Promise<{ error: any }>
  signOut: () => Promise<{ error: any }>
}

const AuthContext = createContext<AuthContextType | undefined>(undefined)

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [user, setUser] = useState<User | null>(null)
  const [dbUser, setDbUser] = useState<DbUser | null>(null)
  const [session, setSession] = useState<Session | null>(null)
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    // Get initial session
    const getInitialSession = async () => {
      const result = await AuthAPI.getCurrentUser()
      setUser(result.user)
      setDbUser(result.dbUser)
      setLoading(false)
    }

    getInitialSession()

    // Listen for auth changes
    const { data: { subscription } } = AuthAPI.onAuthStateChange(async (event, session) => {
      setSession(session)
      setUser(session?.user ?? null)
      
      if (session?.user) {
        const result = await AuthAPI.getCurrentUser()
        setDbUser(result.dbUser)
      } else {
        setDbUser(null)
      }
      
      setLoading(false)
    })

    return () => subscription.unsubscribe()
  }, [])

  const signIn = async (email: string, password: string) => {
    const result = await AuthAPI.signIn(email, password)
    if (result.user && result.dbUser) {
      setUser(result.user)
      setDbUser(result.dbUser)
      setSession(result.session)
    }
    return { error: result.error }
  }

  const signUp = async (email: string, password: string, userData: { username: string }) => {
    const result = await AuthAPI.signUp(email, password, userData)
    if (result.user && result.dbUser) {
      setUser(result.user)
      setDbUser(result.dbUser)
    }
    return { error: result.error }
  }

  const signOut = async () => {
    const result = await AuthAPI.signOut()
    setUser(null)
    setDbUser(null)
    setSession(null)
    return result
  }

  const value = {
    user,
    dbUser,
    session,
    loading,
    signIn,
    signUp,
    signOut
  }

  return (
    <AuthContext.Provider value={value}>
      {children}
    </AuthContext.Provider>
  )
}

export function useAuth() {
  const context = useContext(AuthContext)
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider')
  }
  return context
}
