"use client"

import { useState, useEffect, useRef } from 'react'
import { Server, ArrowUp, Clock, Zap, Play, Square, Shield, Globe, Rocket, TrendingUp, Users, Star, Settings, BarChart3, Power, ExternalLink, Moon, Sun, Heart, Bell, User, LogOut, Menu, X, Plus, Trash2, Activity, Target, Timer, Award, MapPin, Gamepad2, Crown, Eye, Wifi, Database, RefreshCw, Search, Filter, ChevronDown, ChevronUp, Copy, Share2, Bookmark, Terminal, Pause, AlertCircle, CheckCircle, Info, XCircle, Maximize2, Minimize2, RotateCcw } from 'lucide-react'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Progress } from '@/components/ui/progress'
import { Badge } from '@/components/ui/badge'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog'
import { Separator } from '@/components/ui/separator'
import { ScrollArea } from '@/components/ui/scroll-area'
import { Switch } from '@/components/ui/switch'
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar'
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from '@/components/ui/dropdown-menu'
import { toast } from '@/hooks/use-toast'
import { Toaster } from '@/components/ui/toaster'
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, AreaChart, Area, BarChart, Bar, PieChart, Pie, Cell } from 'recharts'

// Import Supabase integration
import { DatabaseAPI, AuthAPI, type DbUser, type DbServer, type DbAnalytics, type DbFavorite } from '@/lib/database'
import { supabase } from '@/lib/supabase'
import type { Session } from '@supabase/supabase-js'

// Enhanced interfaces with Supabase integration
interface User {
  id: string
  username: string
  email: string
  avatar?: string
  joinDate: string
  totalUpvotes: number
  totalServers: number
  plan: 'free' | 'premium' | 'enterprise'
  favoriteServers: string[]
  lastActive: string
  country: string
  verified: boolean
  apiKey?: string
}

interface ServerInfo {
  id: string
  hostname: string
  clients: number
  sv_maxclients: number
  mapname: string
  gametype: string
  upvotePower: number
  country: string
  language: string
  tags: string[]
  resources: string[]
  variables: Record<string, any>
  connectEndPoints: string[]
  ownerProfile: any
  lastSeen: string
  premium: boolean
  burst: boolean
  fallback: boolean
  private: boolean
  localeCountry: string
  hostname_locale: string
  icon: string
  banner: string
  description: string
  website: string
  discord: string
  uptime: number
  avgPlayers: number
  maxPlayersRecord: number
  rating: number
  reviews: number
  categories: string[]
  mods: string[]
  scripts: string[]
  onesync: boolean
  enforceGameBuild: string
  pureLevel: number
  licenseKeyToken: string
  lastPing?: number
  responseTime?: number
  serverLoad?: number
  memoryUsage?: number
  cpuUsage?: number
}

interface AnalyticsData {
  id: string
  userId: string
  serverId: string
  timestamp: string
  upvotes: number
  speed: number
  duration: number
  success: boolean
  type: 'upvote' | 'powerboost'
}

interface FavoriteServer {
  id: string
  serverId: string
  userId: string
  name: string
  address: string
  players: number
  maxPlayers: number
  addedAt: string
  lastChecked: string
  status: 'online' | 'offline'
  ping: number
}

interface ConsoleMessage {
  id: number
  message: string
  type: 'info' | 'success' | 'error' | 'warning' | 'system' | 'network' | 'process'
  timestamp: string
  category: string
  details?: any
  serverId?: string
  userId?: string
}

// Supabase-integrated API functions
const api = {
  users: {
    async authenticate(email: string, password: string): Promise<User | null> {
      try {
        const result = await AuthAPI.signIn(email, password)
        
        if (result.error || !result.dbUser) {
          console.error('Authentication failed:', result.error)
          return null
        }

        // Convert DbUser to User interface
        return {
          id: result.dbUser.id,
          username: result.dbUser.username,
          email: result.dbUser.email,
          avatar: result.dbUser.avatar || undefined,
          joinDate: result.dbUser.join_date,
          totalUpvotes: Number(result.dbUser.total_upvotes),
          totalServers: result.dbUser.total_servers,
          plan: result.dbUser.plan,
          favoriteServers: result.dbUser.favorite_servers,
          lastActive: result.dbUser.last_active,
          country: result.dbUser.country,
          verified: result.dbUser.verified,
          apiKey: result.dbUser.api_key || undefined
        }
      } catch (error) {
        console.error('Authentication error:', error)
        return null
      }
    },
    
    async register(userData: { username: string; email: string; password: string }): Promise<User | null> {
      try {
        const result = await AuthAPI.signUp(userData.email, userData.password, {
          username: userData.username
        })
        
        if (result.error || !result.dbUser) {
          console.error('Registration failed:', result.error)
          return null
        }

        // Convert DbUser to User interface
        return {
          id: result.dbUser.id,
          username: result.dbUser.username,
          email: result.dbUser.email,
          avatar: result.dbUser.avatar || undefined,
          joinDate: result.dbUser.join_date,
          totalUpvotes: Number(result.dbUser.total_upvotes),
          totalServers: result.dbUser.total_servers,
          plan: result.dbUser.plan,
          favoriteServers: result.dbUser.favorite_servers,
          lastActive: result.dbUser.last_active,
          country: result.dbUser.country,
          verified: result.dbUser.verified,
          apiKey: result.dbUser.api_key || undefined
        }
      } catch (error) {
        console.error('Registration error:', error)
        return null
      }
    },
    
    async update(id: string, data: Partial<User>): Promise<User | null> {
      try {
        const dbUser = await DatabaseAPI.updateUser(id, {
          username: data.username,
          email: data.email,
          avatar: data.avatar,
          total_upvotes: data.totalUpvotes ? BigInt(data.totalUpvotes) : undefined,
          total_servers: data.totalServers,
          plan: data.plan,
          favorite_servers: data.favoriteServers,
          last_active: data.lastActive,
          country: data.country,
          verified: data.verified,
          api_key: data.apiKey
        })
        
        if (!dbUser) return null

        return {
          id: dbUser.id,
          username: dbUser.username,
          email: dbUser.email,
          avatar: dbUser.avatar || undefined,
          joinDate: dbUser.join_date,
          totalUpvotes: Number(dbUser.total_upvotes),
          totalServers: dbUser.total_servers,
          plan: dbUser.plan,
          favoriteServers: dbUser.favorite_servers,
          lastActive: dbUser.last_active,
          country: dbUser.country,
          verified: dbUser.verified,
          apiKey: dbUser.api_key || undefined
        }
      } catch (error) {
        console.error('User update error:', error)
        return null
      }
    },

    async getCurrentUser(): Promise<User | null> {
      try {
        const result = await AuthAPI.getCurrentUser()
        
        if (result.error || !result.dbUser) {
          return null
        }

        return {
          id: result.dbUser.id,
          username: result.dbUser.username,
          email: result.dbUser.email,
          avatar: result.dbUser.avatar || undefined,
          joinDate: result.dbUser.join_date,
          totalUpvotes: Number(result.dbUser.total_upvotes),
          totalServers: result.dbUser.total_servers,
          plan: result.dbUser.plan,
          favoriteServers: result.dbUser.favorite_servers,
          lastActive: result.dbUser.last_active,
          country: result.dbUser.country,
          verified: result.dbUser.verified,
          apiKey: result.dbUser.api_key || undefined
        }
      } catch (error) {
        console.error('Get current user error:', error)
        return null
      }
    }
  },
  
  servers: {
    async getInfo(address: string): Promise<ServerInfo | null> {
      try {
        let serverIdentifier = address
        if (address.includes('cfx.re/join/')) {
          serverIdentifier = address.split('cfx.re/join/')[1]
        }
        
        // Check if server exists in database first
        const dbServer = await DatabaseAPI.getServer(serverIdentifier)
        
        // Try to fetch from FiveM API
        const endpoints = [
          `https://servers-frontend.fivem.net/api/servers/single/${serverIdentifier}`,
          `https://servers.fivem.net/api/servers/single/${serverIdentifier}`,
          `https://lambda.fivem.net/api/servers/single/${serverIdentifier}`
        ]
        
        let serverData: any = null
        
        for (const endpoint of endpoints) {
          try {
            const response = await fetch(endpoint, {
              headers: {
                'User-Agent': 'FiveM-UDG-Panel/5.0',
                'Accept': 'application/json'
              }
            })
            
            if (response.ok) {
              const data = await response.json()
              serverData = data.Data || data.data || data
              break
            }
          } catch (e) {
            continue
          }
        }
        
        if (!serverData && !dbServer) {
          return null
        }
        
        // Prepare server info for database
        const serverInfo: ServerInfo = {
          id: serverIdentifier,
          hostname: serverData?.hostname || dbServer?.hostname || 'Unknown Server',
          clients: serverData?.clients || dbServer?.clients || 0,
          sv_maxclients: serverData?.sv_maxclients || dbServer?.sv_maxclients || 32,
          mapname: serverData?.mapname || dbServer?.mapname || 'Unknown',
          gametype: serverData?.gametype || dbServer?.gametype || 'Unknown',
          upvotePower: serverData?.upvotePower || dbServer?.upvote_power || Math.floor(Math.random() * 100) + 50,
          country: serverData?.localeCountry || serverData?.country || dbServer?.country || 'Unknown',
          language: serverData?.locale || dbServer?.language || 'en',
          tags: serverData?.tags || dbServer?.tags || [],
          resources: serverData?.resources || dbServer?.resources || [],
          variables: serverData?.vars || dbServer?.variables || {},
          connectEndPoints: serverData?.connectEndPoints || dbServer?.connect_endpoints || [],
          ownerProfile: serverData?.ownerProfile || dbServer?.owner_profile || null,
          lastSeen: serverData?.lastSeen || dbServer?.last_seen || new Date().toISOString(),
          premium: serverData?.premium || dbServer?.premium || false,
          burst: serverData?.burst || dbServer?.burst || false,
          fallback: serverData?.fallback || dbServer?.fallback || false,
          private: serverData?.private || dbServer?.private || false,
          localeCountry: serverData?.localeCountry || dbServer?.locale_country || 'Unknown',
          hostname_locale: serverData?.hostname || dbServer?.hostname_locale || '',
          icon: serverData?.icon || dbServer?.icon || '',
          banner: serverData?.banner || dbServer?.banner || '',
          description: serverData?.vars?.sv_projectDesc || dbServer?.description || '',
          website: serverData?.vars?.sv_projectName || dbServer?.website || '',
          discord: serverData?.vars?.Discord || dbServer?.discord || '',
          uptime: dbServer?.uptime || Math.floor(Math.random() * 100),
          avgPlayers: dbServer?.avg_players || Math.floor((serverData?.clients || 0) * 0.7),
          maxPlayersRecord: dbServer?.max_players_record || Math.floor((serverData?.sv_maxclients || 32) * 1.2),
          rating: dbServer?.rating || 4.2 + Math.random() * 0.8,
          reviews: dbServer?.reviews || Math.floor(Math.random() * 500) + 50,
          categories: serverData?.tags?.slice(0, 3) || dbServer?.categories || ['Roleplay'],
          mods: serverData?.resources?.slice(0, 5) || dbServer?.mods || [],
          scripts: serverData?.resources?.slice(5, 10) || dbServer?.scripts || [],
          onesync: serverData?.vars?.onesync_enabled === 'true' || dbServer?.onesync || false,
          enforceGameBuild: serverData?.vars?.sv_enforceGameBuild || dbServer?.enforce_game_build || 'latest',
          pureLevel: parseInt(serverData?.vars?.sv_pureLevel) || dbServer?.pure_level || 0,
          licenseKeyToken: serverData?.vars?.sv_licenseKeyToken || dbServer?.license_key_token || '',
          lastPing: dbServer?.last_ping || Math.floor(Math.random() * 100) + 20,
          responseTime: dbServer?.response_time || Math.floor(Math.random() * 50) + 10,
          serverLoad: dbServer?.server_load || Math.floor(Math.random() * 80) + 10,
          memoryUsage: dbServer?.memory_usage || Math.floor(Math.random() * 70) + 20,
          cpuUsage: dbServer?.cpu_usage || Math.floor(Math.random() * 60) + 15
        }
        
        // Update/insert server in database
        await DatabaseAPI.upsertServer({
          id: serverInfo.id,
          hostname: serverInfo.hostname,
          clients: serverInfo.clients,
          sv_maxclients: serverInfo.sv_maxclients,
          mapname: serverInfo.mapname,
          gametype: serverInfo.gametype,
          upvote_power: serverInfo.upvotePower,
          country: serverInfo.country,
          language: serverInfo.language,
          tags: serverInfo.tags,
          resources: serverInfo.resources,
          variables: serverInfo.variables,
          connect_endpoints: serverInfo.connectEndPoints,
          owner_profile: serverInfo.ownerProfile,
          last_seen: serverInfo.lastSeen,
          premium: serverInfo.premium,
          burst: serverInfo.burst,
          fallback: serverInfo.fallback,
          private: serverInfo.private,
          locale_country: serverInfo.localeCountry,
          hostname_locale: serverInfo.hostname_locale,
          icon: serverInfo.icon,
          banner: serverInfo.banner,
          description: serverInfo.description,
          website: serverInfo.website,
          discord: serverInfo.discord,
          uptime: serverInfo.uptime,
          avg_players: serverInfo.avgPlayers,
          max_players_record: serverInfo.maxPlayersRecord,
          rating: serverInfo.rating,
          reviews: serverInfo.reviews,
          categories: serverInfo.categories,
          mods: serverInfo.mods,
          scripts: serverInfo.scripts,
          onesync: serverInfo.onesync,
          enforce_game_build: serverInfo.enforceGameBuild,
          pure_level: serverInfo.pureLevel,
          license_key_token: serverInfo.licenseKeyToken,
          last_ping: serverInfo.lastPing,
          response_time: serverInfo.responseTime,
          server_load: serverInfo.serverLoad,
          memory_usage: serverInfo.memoryUsage,
          cpu_usage: serverInfo.cpuUsage
        })
        
        return serverInfo
      } catch (error) {
        console.error('Server info error:', error)
        return null
      }
    },
    
    async sendUpvotes(serverId: string, amount: number, userId: string): Promise<boolean> {
      try {
        // Create analytics record
        await DatabaseAPI.createAnalytics({
          user_id: userId,
          server_id: serverId,
          upvotes: amount,
          speed: Math.floor(amount / 30), // Assuming 30 second duration
          duration: 30,
          success: true,
          type: 'upvote'
        })
        
        return true
      } catch (error) {
        console.error('Send upvotes error:', error)
        return false
      }
    }
  },
  
  analytics: {
    async create(data: Partial<AnalyticsData>): Promise<AnalyticsData | null> {
      try {
        const dbAnalytics = await DatabaseAPI.createAnalytics({
          user_id: data.userId!,
          server_id: data.serverId!,
          upvotes: data.upvotes || 0,
          speed: data.speed || 0,
          duration: data.duration || 0,
          success: data.success || true,
          type: data.type || 'upvote'
        })
        
        if (!dbAnalytics) return null

        return {
          id: dbAnalytics.id,
          userId: dbAnalytics.user_id,
          serverId: dbAnalytics.server_id,
          timestamp: dbAnalytics.timestamp,
          upvotes: dbAnalytics.upvotes,
          speed: dbAnalytics.speed,
          duration: dbAnalytics.duration,
          success: dbAnalytics.success,
          type: dbAnalytics.type
        }
      } catch (error) {
        console.error('Analytics create error:', error)
        return null
      }
    },
    
    async getByUser(userId: string): Promise<AnalyticsData[]> {
      try {
        const dbAnalytics = await DatabaseAPI.getUserAnalytics(userId)
        
        return dbAnalytics.map(item => ({
          id: item.id,
          userId: item.user_id,
          serverId: item.server_id,
          timestamp: item.timestamp,
          upvotes: item.upvotes,
          speed: item.speed,
          duration: item.duration,
          success: item.success,
          type: item.type
        }))
      } catch (error) {
        console.error('Analytics fetch error:', error)
        return []
      }
    }
  },
  
  favorites: {
    async create(data: Partial<FavoriteServer>): Promise<FavoriteServer | null> {
      try {
        const dbFavorite = await DatabaseAPI.createFavorite({
          server_id: data.serverId!,
          user_id: data.userId!,
          name: data.name!,
          address: data.address!,
          players: data.players || 0,
          max_players: data.maxPlayers || 32,
          ping: data.ping || 0
        })
        
        if (!dbFavorite) return null

        return {
          id: dbFavorite.id,
          serverId: dbFavorite.server_id,
          userId: dbFavorite.user_id,
          name: dbFavorite.name,
          address: dbFavorite.address,
          players: dbFavorite.players,
          maxPlayers: dbFavorite.max_players,
          addedAt: dbFavorite.added_at,
          lastChecked: dbFavorite.last_checked,
          status: dbFavorite.status,
          ping: dbFavorite.ping
        }
      } catch (error) {
        console.error('Favorites create error:', error)
        return null
      }
    },
    
    async getByUser(userId: string): Promise<FavoriteServer[]> {
      try {
        const dbFavorites = await DatabaseAPI.getUserFavorites(userId)
        
        return dbFavorites.map(item => ({
          id: item.id,
          serverId: item.server_id,
          userId: item.user_id,
          name: item.name,
          address: item.address,
          players: item.players,
          maxPlayers: item.max_players,
          addedAt: item.added_at,
          lastChecked: item.last_checked,
          status: item.status,
          ping: item.ping
        }))
      } catch (error) {
        console.error('Favorites fetch error:', error)
        return []
      }
    },
    
    async delete(id: string): Promise<boolean> {
      try {
        return await DatabaseAPI.deleteFavorite(id)
      } catch (error) {
        console.error('Favorites delete error:', error)
        return false
      }
    }
  }
}

export default function FiveMUDGUltimate() {
  // Theme and UI state
  const [isDarkMode, setIsDarkMode] = useState(true)
  const [showMobileMenu, setShowMobileMenu] = useState(false)
  
  // Authentication state
  const [isAuthenticated, setIsAuthenticated] = useState(false)
  const [showAuthModal, setShowAuthModal] = useState(false)
  const [authMode, setAuthMode] = useState<'login' | 'register'>('login')
  const [user, setUser] = useState<User | null>(null)
  const [session, setSession] = useState<Session | null>(null)
  
  // Main application state
  const [serverInput, setServerInput] = useState('')
  const [upvotesInput, setUpvotesInput] = useState('')
  const [powerBoostDuration, setPowerBoostDuration] = useState('')
  const [powerBoostType, setPowerBoostType] = useState('performance')
  const [serverStatus, setServerStatus] = useState<'idle' | 'validating' | 'valid' | 'invalid'>('idle')
  const [serverInfo, setServerInfo] = useState<ServerInfo | null>(null)
  const [activeTab, setActiveTab] = useState('upvotes')
  const [favoriteServers, setFavoriteServers] = useState<FavoriteServer[]>([])
  const [analyticsData, setAnalyticsData] = useState<AnalyticsData[]>([])
  const [realtimeAnalytics, setRealtimeAnalytics] = useState<any[]>([])
  const [showServerDetails, setShowServerDetails] = useState(false)
  const [isRefreshing, setIsRefreshing] = useState(false)
  
  // Resizable panels state
  const [browserPanelSize, setBrowserPanelSize] = useState({ width: 800, height: 800 })
  const [consolePanelSize, setConsolePanelSize] = useState({ height: 400 })
  const [isBrowserMaximized, setIsBrowserMaximized] = useState(false)
  const [isConsoleMaximized, setIsConsoleMaximized] = useState(false)
  
  // Process state
  const [state, setState] = useState({
    progress: 0,
    isRunning: false,
    startTime: null as number | null,
    totalUpvotes: 0,
    targetUpvotes: 0,
    upvotesPerSecond: 10,
    udgKeyVerified: false,
    udgKeyAttempts: 0,
    maxUdgAttempts: 3
  })
  
  // UDG Modal state
  const [showUdgModal, setShowUdgModal] = useState(false)
  const [udgKeyInput, setUdgKeyInput] = useState('')
  const [udgError, setUdgError] = useState('')
  
  // Enhanced Console state
  const [consoleMessages, setConsoleMessages] = useState<ConsoleMessage[]>([])
  const [consoleFilter, setConsoleFilter] = useState<string>('all')
  const [consoleSearch, setConsoleSearch] = useState<string>('')
  const [isConsolePaused, setIsConsolePaused] = useState(false)
  const [consoleAutoScroll, setConsoleAutoScroll] = useState(true)
  const messageIdRef = useRef(0)
  const consoleRef = useRef<HTMLDivElement>(null)
  
  // Real-time monitoring state
  const [serverMetrics, setServerMetrics] = useState({
    ping: 0,
    responseTime: 0,
    serverLoad: 0,
    memoryUsage: 0,
    cpuUsage: 0,
    networkStatus: 'stable' as 'stable' | 'unstable' | 'offline'
  })
  
  // Intervals and subscriptions
  const intervalRef = useRef<NodeJS.Timeout | null>(null)
  const messageIntervalRef = useRef<NodeJS.Timeout | null>(null)
  const analyticsIntervalRef = useRef<NodeJS.Timeout | null>(null)
  const serverMonitoringRef = useRef<NodeJS.Timeout | null>(null)
  const consoleUpdateRef = useRef<NodeJS.Timeout | null>(null)
  const analyticsSubscriptionRef = useRef<any>(null)
  const serverSubscriptionRef = useRef<any>(null)

  // Enhanced console logging with real-time sync
  const logToConsole = (
    message: string, 
    type: 'info' | 'success' | 'error' | 'warning' | 'system' | 'network' | 'process' = 'info',
    category: string = 'SYSTEM',
    details?: any
  ) => {
    if (isConsolePaused) return
    
    const timestamp = new Date().toLocaleTimeString('en-US', { 
      hour12: false, 
      hour: '2-digit', 
      minute: '2-digit', 
      second: '2-digit',
      fractionalSecondDigits: 3
    })
    
    const newMessage: ConsoleMessage = {
      id: messageIdRef.current++,
      message,
      type,
      timestamp,
      category,
      details,
      serverId: serverInfo?.id,
      userId: user?.id
    }
    
    setConsoleMessages(prev => {
      const updated = [...prev.slice(-199), newMessage] // Keep last 200 messages
      return updated
    })
    
    // Auto-scroll to bottom if enabled
    if (consoleAutoScroll && consoleRef.current) {
      setTimeout(() => {
        if (consoleRef.current) {
          consoleRef.current.scrollTop = consoleRef.current.scrollHeight
        }
      }, 100)
    }
    
    // Show toast for important messages
    if (type === 'error' || type === 'success') {
      toast({
        title: type === 'error' ? "Error" : "Success",
        description: message,
        variant: type === 'error' ? "destructive" : "default",
      })
    }
  }

  // Initialize Supabase auth listener
  useEffect(() => {
    // Check current session
    const initializeAuth = async () => {
      const currentUser = await api.users.getCurrentUser()
      if (currentUser) {
        setUser(currentUser)
        setIsAuthenticated(true)
        
        // Load user data
        const favorites = await api.favorites.getByUser(currentUser.id)
        const analytics = await api.analytics.getByUser(currentUser.id)
        setFavoriteServers(favorites)
        setAnalyticsData(analytics)
        
        logToConsole(`🔄 Session restored for ${currentUser.username}`, 'success', 'AUTH')
        logToConsole(`📚 Loaded ${favorites.length} favorites and ${analytics.length} analytics`, 'info', 'DATA')
      }
    }

    initializeAuth()

    // Listen for auth changes
    const { data: { subscription } } = AuthAPI.onAuthStateChange(async (event, session) => {
      setSession(session)
      
      if (event === 'SIGNED_IN' && session?.user) {
        const currentUser = await api.users.getCurrentUser()
        if (currentUser) {
          setUser(currentUser)
          setIsAuthenticated(true)
          logToConsole(`✅ User signed in: ${currentUser.username}`, 'success', 'AUTH')
        }
      } else if (event === 'SIGNED_OUT') {
        setUser(null)
        setIsAuthenticated(false)
        setFavoriteServers([])
        setAnalyticsData([])
        logToConsole(`👋 User signed out`, 'info', 'AUTH')
      }
    })

    return () => {
      subscription.unsubscribe()
    }
  }, [])

  // Real-time subscriptions
  useEffect(() => {
    if (user) {
      // Subscribe to user analytics
      analyticsSubscriptionRef.current = DatabaseAPI.subscribeToUserAnalytics(
        user.id,
        (payload) => {
          logToConsole(`📊 Real-time analytics update received`, 'info', 'REALTIME')
          // Refresh analytics data
          api.analytics.getByUser(user.id).then(setAnalyticsData)
        }
      )
    }

    if (serverInfo) {
      // Subscribe to server updates
      serverSubscriptionRef.current = DatabaseAPI.subscribeToServerUpdates(
        serverInfo.id,
        (payload) => {
          logToConsole(`🖥️ Server metrics updated in real-time`, 'info', 'REALTIME')
          // Update server metrics
          if (payload.new) {
            setServerMetrics(prev => ({
              ...prev,
              ping: payload.new.last_ping || prev.ping,
              serverLoad: payload.new.server_load || prev.serverLoad,
              memoryUsage: payload.new.memory_usage || prev.memoryUsage,
              cpuUsage: payload.new.cpu_usage || prev.cpuUsage
            }))
          }
        }
      )
    }

    return () => {
      if (analyticsSubscriptionRef.current) {
        supabase.removeChannel(analyticsSubscriptionRef.current)
      }
      if (serverSubscriptionRef.current) {
        supabase.removeChannel(serverSubscriptionRef.current)
      }
    }
  }, [user, serverInfo])

  // Real-time server monitoring with database updates
  const startServerMonitoring = () => {
    if (serverMonitoringRef.current) clearInterval(serverMonitoringRef.current)
    
    serverMonitoringRef.current = setInterval(async () => {
      if (serverInfo) {
        // Simulate real-time server metrics
        const newMetrics = {
          ping: Math.floor(Math.random() * 50) + (serverInfo.lastPing || 50),
          responseTime: Math.floor(Math.random() * 30) + 10,
          serverLoad: Math.floor(Math.random() * 20) + (serverInfo.serverLoad || 50),
          memoryUsage: Math.floor(Math.random() * 15) + (serverInfo.memoryUsage || 45),
          cpuUsage: Math.floor(Math.random() * 25) + (serverInfo.cpuUsage || 35),
          networkStatus: Math.random() > 0.95 ? 'unstable' : 'stable' as 'stable' | 'unstable'
        }
        
        setServerMetrics(newMetrics)
        
        // Update database with new metrics
        await DatabaseAPI.updateServerMetrics(serverInfo.id, {
          clients: serverInfo.clients,
          last_ping: newMetrics.ping,
          server_load: newMetrics.serverLoad,
          memory_usage: newMetrics.memoryUsage,
          cpu_usage: newMetrics.cpuUsage
        })
        
        // Log significant changes
        if (newMetrics.ping > 100) {
          logToConsole(`⚠️ High ping detected: ${newMetrics.ping}ms`, 'warning', 'NETWORK', { ping: newMetrics.ping })
        }
        
        if (newMetrics.serverLoad > 80) {
          logToConsole(`🔥 High server load: ${newMetrics.serverLoad}%`, 'warning', 'PERFORMANCE', { load: newMetrics.serverLoad })
        }
        
        if (newMetrics.networkStatus === 'unstable') {
          logToConsole(`📡 Network instability detected`, 'warning', 'NETWORK')
        }
        
        // Update server info with new metrics
        setServerInfo(prev => prev ? {
          ...prev,
          lastPing: newMetrics.ping,
          responseTime: newMetrics.responseTime,
          serverLoad: newMetrics.serverLoad,
          memoryUsage: newMetrics.memoryUsage,
          cpuUsage: newMetrics.cpuUsage
        } : null)
      }
    }, 2000)
  }

  // Enhanced server validation with comprehensive logging and database integration
  const validateServer = async (address: string): Promise<boolean> => {
    try {
      setServerStatus('validating')
      setIsRefreshing(true)
      
      logToConsole(`🔍 Initiating server validation for: ${address}`, 'info', 'VALIDATION')
      logToConsole(`📡 Resolving server endpoint...`, 'info', 'NETWORK')
      
      // Simulate network delay and steps
      await new Promise(resolve => setTimeout(resolve, 500))
      logToConsole(`🌐 Connecting to FiveM API...`, 'info', 'NETWORK')
      
      await new Promise(resolve => setTimeout(resolve, 300))
      logToConsole(`🔐 Authenticating with CFX services...`, 'info', 'AUTH')
      
      const serverData = await api.servers.getInfo(address)
      
      if (!serverData) {
        logToConsole(`❌ Server not found in CFX registry`, 'error', 'VALIDATION')
        logToConsole(`🔍 Attempting direct connection...`, 'info', 'NETWORK')
        await new Promise(resolve => setTimeout(resolve, 1000))
        logToConsole(`⚠️ Direct connection failed - server offline or invalid`, 'error', 'NETWORK')
        throw new Error('Server not found or offline')
      }
      
      setServerInfo(serverData)
      setServerStatus('valid')
      
      // Detailed validation logging
      logToConsole(`✅ Server validation successful!`, 'success', 'VALIDATION')
      logToConsole(`🏷️ Server Name: ${serverData.hostname}`, 'info', 'SERVER_INFO')
      logToConsole(`👥 Players Online: ${serverData.clients}/${serverData.sv_maxclients}`, 'info', 'SERVER_INFO')
      logToConsole(`🌍 Location: ${serverData.country} (${serverData.language})`, 'info', 'SERVER_INFO')
      logToConsole(`🗺️ Current Map: ${serverData.mapname}`, 'info', 'SERVER_INFO')
      logToConsole(`🎮 Game Mode: ${serverData.gametype}`, 'info', 'SERVER_INFO')
      logToConsole(`⚡ Upvote Power: ${serverData.upvotePower} points`, 'info', 'SERVER_INFO')
      logToConsole(`⭐ Server Rating: ${serverData.rating.toFixed(1)}/5.0 (${serverData.reviews} reviews)`, 'info', 'SERVER_INFO')
      logToConsole(`📊 Server Uptime: ${serverData.uptime}%`, 'info', 'PERFORMANCE')
      logToConsole(`🔧 Resources Loaded: ${serverData.resources.length}`, 'info', 'SERVER_INFO')
      logToConsole(`💾 Server data saved to Supabase database`, 'success', 'DATABASE')
      
      if (serverData.tags.length > 0) {
        logToConsole(`🏷️ Server Tags: ${serverData.tags.slice(0, 5).join(', ')}${serverData.tags.length > 5 ? '...' : ''}`, 'info', 'SERVER_INFO')
      }
      
      if (serverData.premium) {
        logToConsole(`👑 Premium server detected - Enhanced features available`, 'success', 'PREMIUM')
      }
      
      if (serverData.onesync) {
        logToConsole(`🔄 OneSync enabled - Enhanced synchronization active`, 'info', 'FEATURES')
      }
      
      if (serverData.burst) {
        logToConsole(`🚀 Burst mode active - High performance server`, 'info', 'FEATURES')
      }
      
      logToConsole(`🔐 Security Level: Pure Level ${serverData.pureLevel}`, 'info', 'SECURITY')
      logToConsole(`🎯 Connection Endpoints: ${serverData.connectEndPoints.length} available`, 'info', 'NETWORK')
      
      // Start real-time monitoring
      startServerMonitoring()
      logToConsole(`📊 Real-time monitoring initiated with database sync`, 'success', 'MONITORING')
      
      toast({
        title: "Server Validated!",
        description: `${serverData.hostname} is online with ${serverData.clients} players`,
      })
      
      return true
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error'
      setServerStatus('invalid')
      setServerInfo(null)
      logToConsole(`❌ Server validation failed: ${errorMessage}`, 'error', 'VALIDATION', { error: errorMessage })
      logToConsole(`🔧 Troubleshooting: Check server address format`, 'info', 'HELP')
      logToConsole(`💡 Tip: Use format 'cfx.re/join/xxxxx' or 'IP:PORT'`, 'info', 'HELP')
      return false
    } finally {
      setIsRefreshing(false)
    }
  }

  // Supabase-integrated authentication
  const handleLogin = async (email: string, password: string) => {
    try {
      logToConsole(`🔐 Authentication attempt for: ${email}`, 'info', 'AUTH')
      logToConsole(`🌐 Connecting to Supabase authentication...`, 'info', 'AUTH')
      
      const userData = await api.users.authenticate(email, password)
      
      if (userData) {
        setUser(userData)
        setIsAuthenticated(true)
        setShowAuthModal(false)
        
        logToConsole(`✅ Authentication successful!`, 'success', 'AUTH')
        logToConsole(`👤 Welcome back, ${userData.username}!`, 'success', 'USER')
        logToConsole(`📊 Account Plan: ${userData.plan.toUpperCase()}`, 'info', 'USER')
        logToConsole(`🎯 Total Upvotes: ${userData.totalUpvotes.toLocaleString()}`, 'info', 'STATS')
        logToConsole(`🏆 Servers Managed: ${userData.totalServers}`, 'info', 'STATS')
        logToConsole(`🌍 Location: ${userData.country}`, 'info', 'USER')
        logToConsole(`📅 Member Since: ${new Date(userData.joinDate).toLocaleDateString()}`, 'info', 'USER')
        logToConsole(`🗄️ Connected to Supabase database`, 'success', 'DATABASE')
        
        if (userData.verified) {
          logToConsole(`✅ Account verified - Full access granted`, 'success', 'USER')
        }
        
        // Load user data
        const favorites = await api.favorites.getByUser(userData.id)
        const analytics = await api.analytics.getByUser(userData.id)
        setFavoriteServers(favorites)
        setAnalyticsData(analytics)
        
        logToConsole(`📚 Loaded ${favorites.length} favorite servers`, 'info', 'DATA')
        logToConsole(`📈 Loaded ${analytics.length} analytics records`, 'info', 'DATA')
        
        toast({
          title: "Login Successful!",
          description: `Welcome back, ${userData.username}!`,
        })
      } else {
        logToConsole(`❌ Authentication failed: Invalid credentials`, 'error', 'AUTH')
        logToConsole(`🔒 Security: Failed login attempt logged`, 'warning', 'SECURITY')
        throw new Error('Invalid credentials')
      }
    } catch (error) {
      logToConsole(`❌ Login failed: ${error instanceof Error ? error.message : 'Unknown error'}`, 'error', 'AUTH')
    }
  }

  const handleRegister = async (username: string, email: string, password: string) => {
    try {
      logToConsole(`📝 Registration attempt for: ${username} (${email})`, 'info', 'AUTH')
      logToConsole(`🔍 Validating user data with Supabase...`, 'info', 'AUTH')
      
      const userData = await api.users.register({ username, email, password })
      
      if (userData) {
        setUser(userData)
        setIsAuthenticated(true)
        setShowAuthModal(false)
        
        logToConsole(`✅ Account created successfully!`, 'success', 'AUTH')
        logToConsole(`🎉 Welcome to FiveM UDG, ${userData.username}!`, 'success', 'USER')
        logToConsole(`📧 Account email: ${userData.email}`, 'info', 'USER')
        logToConsole(`🆔 User ID: ${userData.id}`, 'info', 'SYSTEM')
        logToConsole(`📅 Registration Date: ${new Date().toLocaleDateString()}`, 'info', 'USER')
        logToConsole(`🎯 Starting Plan: FREE`, 'info', 'USER')
        logToConsole(`🗄️ User profile saved to Supabase`, 'success', 'DATABASE')
        logToConsole(`💡 Tip: Upgrade to Premium for enhanced features`, 'info', 'HELP')
        
        toast({
          title: "Account Created!",
          description: `Welcome to FiveM UDG, ${userData.username}!`,
        })
      } else {
        throw new Error('Registration failed')
      }
    } catch (error) {
      logToConsole(`❌ Registration failed: ${error instanceof Error ? error.message : 'Unknown error'}`, 'error', 'AUTH')
    }
  }

  // Enhanced server favoriting with Supabase integration
  const addToFavorites = async () => {
    if (!serverInfo || !user) return
    
    try {
      logToConsole(`❤️ Adding server to favorites: ${serverInfo.hostname}`, 'info', 'FAVORITES')
      
      const favorite = await api.favorites.create({
        serverId: serverInfo.id,
        userId: user.id,
        name: serverInfo.hostname,
        address: serverInput,
        players: serverInfo.clients,
        maxPlayers: serverInfo.sv_maxclients,
        ping: serverInfo.lastPing || 0
      })
      
      if (favorite) {
        setFavoriteServers(prev => [...prev, favorite])
        logToConsole(`✅ Server added to favorites successfully`, 'success', 'FAVORITES')
        logToConsole(`📊 Total favorites: ${favoriteServers.length + 1}`, 'info', 'STATS')
        logToConsole(`💾 Favorite saved to Supabase database`, 'success', 'DATABASE')
        
        toast({
          title: "Server Favorited!",
          description: `${serverInfo.hostname} added to your favorites`,
        })
      }
    } catch (error) {
      logToConsole(`❌ Failed to add server to favorites: ${error instanceof Error ? error.message : 'Unknown error'}`, 'error', 'FAVORITES')
    }
  }

  const removeFromFavorites = async (id: string) => {
    try {
      const server = favoriteServers.find(s => s.id === id)
      logToConsole(`🗑️ Removing server from favorites: ${server?.name || 'Unknown'}`, 'info', 'FAVORITES')
      
      const success = await api.favorites.delete(id)
      
      if (success) {
        setFavoriteServers(prev => prev.filter(server => server.id !== id))
        logToConsole(`✅ Server removed from favorites`, 'success', 'FAVORITES')
        logToConsole(`📊 Total favorites: ${favoriteServers.length - 1}`, 'info', 'STATS')
        logToConsole(`💾 Favorite deleted from Supabase database`, 'success', 'DATABASE')
        
        toast({
          title: "Server Removed",
          description: "Server removed from favorites",
        })
      }
    } catch (error) {
      logToConsole(`❌ Failed to remove server from favorites: ${error instanceof Error ? error.message : 'Unknown error'}`, 'error', 'FAVORITES')
    }
  }

  // Enhanced upvote process with Supabase analytics
  const startProcess = async () => {
    if (!isAuthenticated) {
      logToConsole(`🔐 Authentication required to start upvote process`, 'warning', 'AUTH')
      setShowAuthModal(true)
      return
    }
    
    if (!serverInfo) {
      logToConsole(`❌ No server selected - Please validate a server first`, 'error', 'VALIDATION')
      return
    }
    
    const upvotesAmount = parseInt(upvotesInput)
    if (isNaN(upvotesAmount) || upvotesAmount < 1 || upvotesAmount < 50000) {
      logToConsole(`❌ Invalid upvotes amount: ${upvotesInput} (Range: 1-50,000)`, 'error', 'VALIDATION')
      return
    }
    
    setState(prev => ({
      ...prev,
      targetUpvotes: upvotesAmount,
      isRunning: true,
      startTime: Date.now(),
      progress: 0,
      totalUpvotes: 0,
      udgKeyVerified: false
    }))
    
    // Clear previous analytics
    setRealtimeAnalytics([])
    
    // Comprehensive process initialization logging
    logToConsole(`🚀 Initializing upvote process...`, 'success', 'PROCESS')
    logToConsole(`🎯 Target Server: ${serverInfo.hostname}`, 'info', 'PROCESS')
    logToConsole(`📊 Target Upvotes: ${upvotesAmount.toLocaleString()}`, 'info', 'PROCESS')
    logToConsole(`👤 User: ${user?.username} (${user?.plan.toUpperCase()})`, 'info', 'PROCESS')
    logToConsole(`⚡ Estimated Duration: ${Math.ceil(upvotesAmount / 1000)} seconds`, 'info', 'PROCESS')
    logToConsole(`🔧 Process ID: ${Date.now()}`, 'info', 'SYSTEM')
    logToConsole(`🗄️ Analytics will be saved to Supabase`, 'info', 'DATABASE')
    
    // System checks
    logToConsole(`🔍 Performing system checks...`, 'info', 'SYSTEM')
    await new Promise(resolve => setTimeout(resolve, 500))
    logToConsole(`✅ Memory allocation: OK`, 'success', 'SYSTEM')
    logToConsole(`✅ Network connectivity: OK`, 'success', 'NETWORK')
    logToConsole(`✅ Server response: OK (${serverInfo.lastPing}ms)`, 'success', 'NETWORK')
    logToConsole(`✅ Supabase connection: Active`, 'success', 'DATABASE')
    logToConsole(`✅ Authentication token: Valid`, 'success', 'AUTH')
    
    // Process configuration
    logToConsole(`⚙️ Configuring upvote parameters...`, 'info', 'PROCESS')
    logToConsole(`📡 Target Endpoints: ${serverInfo.connectEndPoints.length}`, 'info', 'NETWORK')
    logToConsole(`🔐 Security Level: ${serverInfo.pureLevel}`, 'info', 'SECURITY')
    logToConsole(`🎮 Server Type: ${serverInfo.premium ? 'Premium' : 'Standard'}`, 'info', 'SERVER_INFO')
    
    toast({
      title: "Process Started!",
      description: `Sending ${upvotesAmount.toLocaleString()} upvotes to ${serverInfo.hostname}`,
    })
    
    // Start process intervals
    intervalRef.current = setInterval(updateProgress, 100)
    analyticsIntervalRef.current = setInterval(updateAnalytics, 1000)
    
    // Enhanced real-time process logging
    messageIntervalRef.current = setInterval(() => {
      const processMessages = [
        { msg: "📡 Establishing secure connection to CFX network...", type: 'info' as const, cat: 'NETWORK' },
        { msg: "🔐 Encrypting upvote packets with AES-256...", type: 'info' as const, cat: 'SECURITY' },
        { msg: "⚡ Processing upvote batch transmission...", type: 'info' as const, cat: 'PROCESS' },
        { msg: "📊 Monitoring server response metrics...", type: 'info' as const, cat: 'MONITORING' },
        { msg: "🛡️ Validating anti-cheat protocols...", type: 'info' as const, cat: 'SECURITY' },
        { msg: "🎯 Optimizing delivery routes to target server...", type: 'info' as const, cat: 'NETWORK' },
        { msg: "📈 Analyzing upvote delivery success rate...", type: 'info' as const, cat: 'ANALYTICS' },
        { msg: "🌐 Synchronizing with FiveM master servers...", type: 'info' as const, cat: 'NETWORK' },
        { msg: "🔄 Balancing load across multiple endpoints...", type: 'info' as const, cat: 'PERFORMANCE' },
        { msg: "⚙️ Adjusting transmission speed based on server load...", type: 'info' as const, cat: 'OPTIMIZATION' },
        { msg: "📡 Checking network latency and stability...", type: 'info' as const, cat: 'NETWORK' },
        { msg: "🎮 Verifying server compatibility and status...", type: 'info' as const, cat: 'VALIDATION' },
        { msg: "💾 Streaming analytics data to Supabase...", type: 'info' as const, cat: 'DATABASE' }
      ]
      
      const randomMessage = processMessages[Math.floor(Math.random() * processMessages.length)]
      logToConsole(randomMessage.msg, randomMessage.type, randomMessage.cat)
      
      // Occasional performance updates
      if (Math.random() > 0.7) {
        const currentSpeed = getCurrentSpeed()
        if (currentSpeed > 0) {
          logToConsole(`⚡ Current Speed: ${currentSpeed} upvotes/sec`, 'info', 'PERFORMANCE')
        }
      }
      
      // Network status updates
      if (Math.random() > 0.8) {
        logToConsole(`📊 Server Load: ${serverMetrics.serverLoad}% | Ping: ${serverMetrics.ping}ms`, 'info', 'MONITORING')
      }
    }, 2500)
  }

  const updateProgress = () => {
    setState(prevState => {
      if (!prevState.isRunning) return prevState
      
      const progressIncrement = (100 / 30000) * 100 // 30 seconds total
      const newProgress = Math.min(prevState.progress + progressIncrement, 100)
      const newTotalUpvotes = Math.floor(prevState.targetUpvotes * (newProgress / 100))
      
      // Progress milestone logging
      const milestones = [10, 25, 50, 75, 85, 90, 95]
      const currentMilestone = Math.floor(newProgress)
      const previousMilestone = Math.floor(prevState.progress)
      
      if (milestones.includes(currentMilestone) && currentMilestone !== previousMilestone) {
        logToConsole(`🎯 Progress Milestone: ${currentMilestone}% completed`, 'success', 'PROGRESS')
        logToConsole(`📊 Upvotes Sent: ${newTotalUpvotes.toLocaleString()}/${prevState.targetUpvotes.toLocaleString()}`, 'info', 'PROGRESS')
      }
      
      // UDG Key check at 85%
      if (newProgress >= 85 && !prevState.udgKeyVerified) {
        setShowUdgModal(true)
        logToConsole(`🔐 UDG Key verification required at 85% completion`, 'warning', 'SECURITY')
        logToConsole(`⏸️ Process paused - Awaiting security verification`, 'warning', 'PROCESS')
        logToConsole(`🛡️ This is a security measure to prevent unauthorized usage`, 'info', 'SECURITY')
        return { ...prevState, progress: newProgress, totalUpvotes: newTotalUpvotes, isRunning: false }
      }
      
      // Completion
      if (newProgress >= 100) {
        if (intervalRef.current) clearInterval(intervalRef.current)
        if (messageIntervalRef.current) clearInterval(messageIntervalRef.current)
        if (analyticsIntervalRef.current) clearInterval(analyticsIntervalRef.current)
        
        // Completion logging
        logToConsole(`🎉 Upvote process completed successfully!`, 'success', 'PROCESS')
        logToConsole(`📊 Final Statistics:`, 'info', 'STATS')
        logToConsole(`  ├─ Total Upvotes: ${newTotalUpvotes.toLocaleString()}`, 'info', 'STATS')
        logToConsole(`  ├─ Average Speed: ${Math.floor(newTotalUpvotes / 30)} upvotes/second`, 'info', 'STATS')
        logToConsole(`  ├─ Duration: 30 seconds`, 'info', 'STATS')
        logToConsole(`  ├─ Success Rate: 100%`, 'info', 'STATS')
        logToConsole(`  └─ Server: ${serverInfo?.hostname}`, 'info', 'STATS')
        
        // Save to Supabase database
        if (user && serverInfo) {
          api.analytics.create({
            userId: user.id,
            serverId: serverInfo.id,
            upvotes: newTotalUpvotes,
            speed: Math.floor(newTotalUpvotes / 30),
            duration: 30,
            success: true,
            type: 'upvote'
          }).then(() => {
            logToConsole(`💾 Analytics data saved to Supabase database`, 'success', 'DATABASE')
          })
          
          // Update user's total upvotes
          api.users.update(user.id, {
            totalUpvotes: user.totalUpvotes + newTotalUpvotes,
            lastActive: new Date().toISOString()
          }).then(() => {
            logToConsole(`👤 User profile updated with new statistics`, 'success', 'USER')
          })
        }
        
        toast({
          title: "Upvotes Complete!",
          description: `Successfully sent ${newTotalUpvotes.toLocaleString()} upvotes!`,
        })
        
        return { ...prevState, progress: 100, totalUpvotes: newTotalUpvotes, isRunning: false }
      }
      
      return { ...prevState, progress: newProgress, totalUpvotes: newTotalUpvotes }
    })
  }

  const updateAnalytics = () => {
    if (state.isRunning && state.startTime) {
      const currentTime = new Date().toLocaleTimeString()
      const elapsedTime = (Date.now() - state.startTime) / 1000
      const currentSpeed = Math.floor(state.totalUpvotes / elapsedTime) || 0
      
      setRealtimeAnalytics(prev => [
        ...prev.slice(-19),
        {
          time: currentTime,
          upvotes: state.totalUpvotes,
          speed: currentSpeed,
          progress: state.progress
        }
      ])
    }
  }

  const stopProcess = () => {
    if (intervalRef.current) clearInterval(intervalRef.current)
    if (messageIntervalRef.current) clearInterval(messageIntervalRef.current)
    if (analyticsIntervalRef.current) clearInterval(analyticsIntervalRef.current)
    
    setState(prev => ({
      ...prev,
      isRunning: false,
      progress: 0,
      totalUpvotes: 0,
      startTime: null
    }))
    
    logToConsole(`⏹️ Process stopped by user`, 'warning', 'PROCESS')
    logToConsole(`📊 Process terminated at ${Math.floor(state.progress)}% completion`, 'info', 'STATS')
    logToConsole(`🔄 System ready for new process`, 'info', 'SYSTEM')
    
    toast({
      title: "Process Stopped",
      description: "Upvote process has been terminated",
      variant: "destructive"
    })
  }

  // Enhanced PowerBoost activation with Supabase logging
  const activatePowerBoost = async () => {
    if (!isAuthenticated) {
      logToConsole(`🔐 Authentication required for PowerBoost activation`, 'warning', 'AUTH')
      setShowAuthModal(true)
      return
    }
    
    if (!serverInfo) {
      logToConsole(`❌ No server selected for PowerBoost activation`, 'error', 'VALIDATION')
      return
    }
    
    const duration = parseInt(powerBoostDuration)
    if (isNaN(duration) || duration < 1 || duration > 168) {
      logToConsole(`❌ Invalid PowerBoost duration: ${powerBoostDuration} (Range: 1-168 hours)`, 'error', 'VALIDATION')
      return
    }
    
    logToConsole(`⚡ Initiating PowerBoost activation...`, 'info', 'POWERBOOST')
    logToConsole(`🎯 Target Server: ${serverInfo.hostname}`, 'info', 'POWERBOOST')
    logToConsole(`⚙️ Boost Type: ${powerBoostType.toUpperCase()}`, 'info', 'POWERBOOST')
    logToConsole(`⏰ Duration: ${duration} hours`, 'info', 'POWERBOOST')
    logToConsole(`👤 User: ${user?.username} (${user?.plan.toUpperCase()})`, 'info', 'POWERBOOST')
    
    // Simulate PowerBoost activation process
    logToConsole(`🔍 Validating server compatibility...`, 'info', 'POWERBOOST')
    await new Promise(resolve => setTimeout(resolve, 800))
    logToConsole(`✅ Server compatibility confirmed`, 'success', 'POWERBOOST')
    
    logToConsole(`💰 Processing payment authorization...`, 'info', 'PAYMENT')
    await new Promise(resolve => setTimeout(resolve, 600))
    logToConsole(`✅ Payment authorized`, 'success', 'PAYMENT')
    
    logToConsole(`🚀 Deploying PowerBoost configuration...`, 'info', 'POWERBOOST')
    await new Promise(resolve => setTimeout(resolve, 400))
    
    setTimeout(() => {
      if (user && serverInfo) {
        api.analytics.create({
          userId: user.id,
          serverId: serverInfo.id,
          upvotes: 0,
          speed: 0,
          duration: duration * 3600,
          success: true,
          type: 'powerboost'
        }).then(() => {
          logToConsole(`💾 PowerBoost analytics saved to Supabase`, 'success', 'DATABASE')
        })
      }
      
      logToConsole(`✅ PowerBoost activated successfully!`, 'success', 'POWERBOOST')
      logToConsole(`🚀 Server visibility increased by 200%`, 'success', 'POWERBOOST')
      logToConsole(`⭐ Priority ranking boost is now active`, 'success', 'POWERBOOST')
      logToConsole(`📈 Enhanced analytics tracking enabled`, 'success', 'POWERBOOST')
      logToConsole(`🛡️ DDoS protection layer activated`, 'success', 'POWERBOOST')
      logToConsole(`⏰ PowerBoost will remain active for ${duration} hours`, 'info', 'POWERBOOST')
      logToConsole(`📅 Expiration: ${new Date(Date.now() + duration * 3600000).toLocaleString()}`, 'info', 'POWERBOOST')
      
      toast({
        title: "PowerBoost Activated!",
        description: `${serverInfo.hostname} boosted for ${duration} hours`,
      })
    }, 1200)
  }

  // Enhanced UDG Key handling with logging
  const handleUdgKeySubmit = () => {
    logToConsole(`🔑 UDG Key verification attempt`, 'info', 'SECURITY')
    
    if (udgKeyInput.trim() === 'UDG-PROD-2024-LIVE') {
      setState(prev => ({ ...prev, udgKeyVerified: true, isRunning: true }))
      setShowUdgModal(false)
      setUdgError('')
      setUdgKeyInput('')
      
      logToConsole(`✅ UDG Key verified successfully`, 'success', 'SECURITY')
      logToConsole(`🔓 Security clearance granted`, 'success', 'SECURITY')
      logToConsole(`▶️ Resuming upvote process...`, 'success', 'PROCESS')
      
      // Resume process
      intervalRef.current = setInterval(updateProgress, 100)
      analyticsIntervalRef.current = setInterval(updateAnalytics, 1000)
    } else {
      setState(prev => {
        const newAttempts = prev.udgKeyAttempts + 1
        const remainingAttempts = prev.maxUdgAttempts - newAttempts
        
        logToConsole(`❌ Invalid UDG Key provided`, 'error', 'SECURITY')
        logToConsole(`🔒 Attempts remaining: ${remainingAttempts}`, 'warning', 'SECURITY')
        
        if (remainingAttempts > 0) {
          setUdgError(`Invalid UDG Key. ${remainingAttempts} attempts remaining.`)
          return { ...prev, udgKeyAttempts: newAttempts }
        } else {
          logToConsole(`🚨 Maximum UDG Key attempts exceeded`, 'error', 'SECURITY')
          logToConsole(`🔒 Security lockout initiated`, 'error', 'SECURITY')
          logToConsole(`⏹️ Process terminated for security reasons`, 'error', 'PROCESS')
          
          setShowUdgModal(false)
          toast({
            title: "Access Denied",
            description: "Maximum UDG Key attempts reached",
            variant: "destructive"
          })
          return {
            ...prev,
            udgKeyAttempts: newAttempts,
            isRunning: false,
            progress: 0,
            totalUpvotes: 0,
            startTime: null
          }
        }
      })
      setUdgKeyInput('')
    }
  }

  // Console filtering and search
  const filteredMessages = consoleMessages.filter(msg => {
    const matchesFilter = consoleFilter === 'all' || msg.type === consoleFilter || msg.category === consoleFilter
    const matchesSearch = consoleSearch === '' || 
      msg.message.toLowerCase().includes(consoleSearch.toLowerCase()) ||
      msg.category.toLowerCase().includes(consoleSearch.toLowerCase())
    return matchesFilter && matchesSearch
  })

  // Console controls
  const clearConsole = () => {
    setConsoleMessages([])
    logToConsole(`🧹 Console cleared by user`, 'info', 'SYSTEM')
  }

  const toggleConsolePause = () => {
    setIsConsolePaused(!isConsolePaused)
    logToConsole(`${!isConsolePaused ? '⏸️ Console paused' : '▶️ Console resumed'}`, 'info', 'SYSTEM')
  }

  const exportConsoleLog = () => {
    const logData = consoleMessages.map(msg => 
      `[${msg.timestamp}] [${msg.category}] [${msg.type.toUpperCase()}] ${msg.message}`
    ).join('\n')
    
    const blob = new Blob([logData], { type: 'text/plain' })
    const url = URL.createObjectURL(blob)
    const a = document.createElement('a')
    a.href = url
    a.download = `fivem-udg-console-${new Date().toISOString().split('T')[0]}.log`
    a.click()
    URL.revokeObjectURL(url)
    
    logToConsole(`📁 Console log exported successfully`, 'success', 'SYSTEM')
  }

  // Panel resize functions
  const toggleBrowserMaximize = () => {
    setIsBrowserMaximized(!isBrowserMaximized)
    if (!isBrowserMaximized) {
      setBrowserPanelSize({ width: window.innerWidth - 100, height: window.innerHeight - 200 })
    } else {
      setBrowserPanelSize({ width: 800, height: 800 })
    }
  }

  const toggleConsoleMaximize = () => {
    setIsConsoleMaximized(!isConsoleMaximized)
    if (!isConsoleMaximized) {
      setConsolePanelSize({ height: window.innerHeight - 300 })
    } else {
      setConsolePanelSize({ height: 400 })
    }
  }

  const resetPanelSizes = () => {
    setBrowserPanelSize({ width: 800, height: 800 })
    setConsolePanelSize({ height: 400 })
    setIsBrowserMaximized(false)
    setIsConsoleMaximized(false)
    logToConsole(`🔄 Panel sizes reset to default`, 'info', 'SYSTEM')
  }

  // Utility functions
  const formatElapsedTime = () => {
    if (!state.startTime) return '00:00'
    const elapsed = Math.floor((Date.now() - state.startTime) / 1000)
    const minutes = Math.floor(elapsed / 60)
    const seconds = elapsed % 60
    return `${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`
  }

  const getCurrentSpeed = () => {
    if (!state.startTime || state.totalUpvotes === 0) return 0
    const elapsedTime = (Date.now() - state.startTime) / 1000
    return Math.floor(state.totalUpvotes / elapsedTime) || 0
  }

  const copyServerAddress = () => {
    navigator.clipboard.writeText(serverInput)
    logToConsole(`📋 Server address copied to clipboard: ${serverInput}`, 'info', 'SYSTEM')
    toast({
      title: "Copied!",
      description: "Server address copied to clipboard",
    })
  }

  // Theme toggle
  const toggleTheme = () => {
    setIsDarkMode(!isDarkMode)
    localStorage.setItem('theme', !isDarkMode ? 'dark' : 'light')
    logToConsole(`🎨 Theme changed to ${!isDarkMode ? 'dark' : 'light'} mode`, 'info', 'SYSTEM')
    toast({
      title: "Theme Changed",
      description: `Switched to ${!isDarkMode ? 'dark' : 'light'} mode`,
    })
  }

  // Initialize with enhanced logging
  useEffect(() => {
    const savedTheme = localStorage.getItem('theme')
    if (savedTheme) {
      setIsDarkMode(savedTheme === 'dark')
    }
    
    // System initialization logging
    logToConsole(`🚀 FiveM UDG V 5.0 Ultimate with Supabase initialized`, 'success', 'SYSTEM')
    logToConsole(`🌐 Version: 5.0.0 Supabase Edition`, 'info', 'SYSTEM')
    logToConsole(`💻 Platform: ${navigator.platform}`, 'info', 'SYSTEM')
    logToConsole(`🌍 User Agent: ${navigator.userAgent.split(' ')[0]}`, 'info', 'SYSTEM')
    logToConsole(`🎨 Theme: ${savedTheme || 'dark'} mode`, 'info', 'SYSTEM')
    logToConsole(`🗄️ Supabase database connected`, 'success', 'DATABASE')
    logToConsole(`🔐 Security protocols initialized`, 'success', 'SECURITY')
    logToConsole(`📊 Analytics engine ready`, 'success', 'ANALYTICS')
    logToConsole(`🔄 Real-time subscriptions active`, 'success', 'REALTIME')
    logToConsole(`🔧 All systems operational`, 'success', 'SYSTEM')
    logToConsole(`ℹ️ Please login to access all features`, 'info', 'AUTH')
    
    // Start console update interval for real-time sync
    consoleUpdateRef.current = setInterval(() => {
      if (serverInfo && !isConsolePaused) {
        // Periodic system status updates
        if (Math.random() > 0.95) {
          logToConsole(`📊 System Status: All services operational`, 'info', 'MONITORING')
        }
      }
    }, 10000)
    
    return () => {
      if (consoleUpdateRef.current) clearInterval(consoleUpdateRef.current)
    }
  }, [])

  // Cleanup
  useEffect(() => {
    return () => {
      if (intervalRef.current) clearInterval(intervalRef.current)
      if (messageIntervalRef.current) clearInterval(messageIntervalRef.current)
      if (analyticsIntervalRef.current) clearInterval(analyticsIntervalRef.current)
      if (serverMonitoringRef.current) clearInterval(serverMonitoringRef.current)
      if (consoleUpdateRef.current) clearInterval(consoleUpdateRef.current)
    }
  }, [])

  return (
    <div className={`min-h-screen transition-all duration-500 ${
      isDarkMode 
        ? 'bg-gradient-to-br from-slate-950 via-purple-950 to-slate-950' 
        : 'bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50'
    }`}>
      {/* Animated Background Elements */}
      <div className="fixed inset-0 overflow-hidden pointer-events-none">
        <div className={`absolute -top-40 -right-40 w-80 h-80 rounded-full opacity-20 blur-3xl animate-pulse ${
          isDarkMode ? 'bg-purple-500' : 'bg-blue-400'
        }`} />
        <div className={`absolute -bottom-40 -left-40 w-80 h-80 rounded-full opacity-20 blur-3xl animate-pulse delay-1000 ${
          isDarkMode ? 'bg-blue-500' : 'bg-purple-400'
        }`} />
      </div>

      <div className="relative z-10 container mx-auto p-4 max-w-[1600px]">
        {/* Enhanced Header */}
        <div className={`rounded-3xl p-8 mb-8 backdrop-blur-xl border-2 shadow-2xl ${
          isDarkMode 
            ? 'bg-slate-900/40 border-slate-700/50' 
            : 'bg-white/60 border-white/30'
        }`}>
          <div className="flex items-center justify-between mb-6">
            <div className="flex items-center gap-6">
              <div className="relative">
                <div className="absolute inset-0 bg-gradient-to-r from-blue-500 to-purple-600 rounded-2xl blur-lg opacity-75 animate-pulse" />
                <div className="relative p-4 bg-gradient-to-r from-blue-500 to-purple-600 rounded-2xl">
                  <Rocket className="w-10 h-10 text-white" />
                </div>
              </div>
              <div>
                <h1 className="text-5xl font-black bg-gradient-to-r from-blue-400 via-purple-400 to-pink-400 bg-clip-text text-transparent">
                  FiveM UDG V 5.0
                </h1>
                <p className="text-2xl font-bold bg-gradient-to-r from-emerald-400 to-cyan-400 bg-clip-text text-transparent">
                  SUPABASE EDITION
                </p>
                <p className={`text-lg mt-1 ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                  Professional Server Enhancement Suite with Real-time Database
                </p>
              </div>
            </div>
            
            <div className="flex items-center gap-4">
              {/* Discord Button */}
              <Button
                onClick={() => window.open('https://discord.gg/A3dEYdqnyK', '_blank')}
                className="bg-gradient-to-r from-indigo-500 to-purple-600 hover:from-indigo-600 hover:to-purple-700 px-6 py-3 rounded-2xl font-bold"
              >
                <svg className="w-5 h-5 mr-2" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M20.317 4.37a19.791 19.791 0 0 0-4.885-1.515a.074.074 0 0 0-.079.037c-.21.375-.444.864-.608 1.25a18.27 18.27 0 0 0-5.487 0a12.64 12.64 0 0 0-.617-1.25a.077.077 0 0 0-.079-.037A19.736 19.736 0 0 0 3.677 4.37a.07.07 0 0 0-.032.027C.533 9.046-.32 13.58.099 18.057a.082.082 0 0 0 .031.057a19.9 19.9 0 0 0 5.993 3.03a.078.078 0 0 0 .084-.028a14.09 14.09 0 0 0 1.226-1.994a.076.076 0 0 0-.041-.106a13.107 13.107 0 0 1-1.872-.892a.077.077 0 0 1-.008-.128a10.2 10.2 0 0 0 .372-.292a.074.074 0 0 1 .077-.01c3.928 1.793 8.18 1.793 12.062 0a.074.074 0 0 1 .078.01c.12.098.246.198.373.292a.077.077 0 0 1-.006.127a12.299 12.299 0 0 1-1.873.892a.077.077 0 0 0-.041.107c.36.698.772 1.362 1.225 1.993a.076.076 0 0 0 .084.028a19.839 19.839 0 0 0 6.002-3.03a.077.077 0 0 0 .032-.054c.5-5.177-.838-9.674-3.549-13.66a.061.061 0 0 0-.031-.03zM8.02 15.33c-1.183 0-2.157-1.085-2.157-2.419c0-1.333.956-2.419 2.157-2.419c1.21 0 2.176 1.096 2.157 2.42c0 1.333-.956 2.418-2.157 2.418zm7.975 0c-1.183 0-2.157-1.085-2.157-2.419c0-1.333.955-2.419 2.157-2.419c1.21 0 2.176 1.096 2.157 2.42c0 1.333-.946 2.418-2.157 2.418z"/>
                </svg>
                Discord
              </Button>
              
              {/* Panel Reset Button */}
              <Button
                onClick={resetPanelSizes}
                variant="outline"
                className="px-4 py-3 rounded-2xl"
                title="Reset Panel Sizes"
              >
                <RotateCcw className="w-5 h-5" />
              </Button>
              
              {/* Theme Toggle */}
              <div className={`flex items-center gap-3 p-3 rounded-2xl backdrop-blur-sm ${
                isDarkMode ? 'bg-slate-800/50' : 'bg-white/50'
              }`}>
                <Sun className={`w-5 h-5 ${isDarkMode ? 'text-slate-400' : 'text-yellow-500'}`} />
                <Switch checked={isDarkMode} onCheckedChange={toggleTheme} />
                <Moon className={`w-5 h-5 ${isDarkMode ? 'text-blue-400' : 'text-slate-400'}`} />
              </div>
              
              {/* User Menu */}
              {isAuthenticated && user ? (
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="ghost" className={`flex items-center gap-3 p-4 rounded-2xl backdrop-blur-sm ${
                      isDarkMode ? 'bg-slate-800/50 hover:bg-slate-700/50' : 'bg-white/50 hover:bg-white/70'
                    }`}>
                      <Avatar className="w-10 h-10 ring-2 ring-blue-400">
                        <AvatarImage src={user.avatar || "/placeholder.svg"} />
                        <AvatarFallback className="bg-gradient-to-r from-blue-500 to-purple-600 text-white">
                          {user.username[0].toUpperCase()}
                        </AvatarFallback>
                      </Avatar>
                      <div className="text-left">
                        <p className={`font-bold ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                          {user.username}
                        </p>
                        <Badge className={`text-xs ${
                          user.plan === 'enterprise' ? 'bg-purple-500/20 text-purple-300' :
                          user.plan === 'premium' ? 'bg-blue-500/20 text-blue-300' :
                          'bg-slate-500/20 text-slate-300'
                        }`}>
                          {user.plan.toUpperCase()}
                        </Badge>
                      </div>
                      <ChevronDown className="w-4 h-4" />
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="end" className={`w-64 ${
                    isDarkMode ? 'bg-slate-800 border-slate-700' : 'bg-white border-slate-200'
                  }`}>
                    <div className="p-4 border-b border-slate-200 dark:border-slate-700">
                      <div className="flex items-center gap-3">
                        <Avatar className="w-12 h-12">
                          <AvatarImage src={user.avatar || "/placeholder.svg"} />
                          <AvatarFallback>{user.username[0].toUpperCase()}</AvatarFallback>
                        </Avatar>
                        <div>
                          <p className="font-bold">{user.username}</p>
                          <p className="text-sm text-slate-500">{user.email}</p>
                          <p className="text-xs text-slate-400">
                            {user.totalUpvotes.toLocaleString()} total upvotes
                          </p>
                        </div>
                      </div>
                    </div>
                    <DropdownMenuItem>
                      <User className="w-4 h-4 mr-2" />
                      Profile Settings
                    </DropdownMenuItem>
                    <DropdownMenuItem>
                      <Database className="w-4 h-4 mr-2" />
                      Analytics Dashboard
                    </DropdownMenuItem>
                    <DropdownMenuItem>
                      <Settings className="w-4 h-4 mr-2" />
                      Preferences
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={async () => {
                      await AuthAPI.signOut()
                      setIsAuthenticated(false)
                      setUser(null)
                    }}>
                      <LogOut className="w-4 h-4 mr-2" />
                      Logout
                    </DropdownMenuItem>
                  </DropdownMenuContent>
                </DropdownMenu>
              ) : (
                <Button 
                  onClick={() => setShowAuthModal(true)} 
                  className="bg-gradient-to-r from-blue-500 to-purple-600 hover:from-blue-600 hover:to-purple-700 px-6 py-3 rounded-2xl font-bold"
                >
                  <User className="w-5 h-5 mr-2" />
                  Login / Register
                </Button>
              )}
            </div>
          </div>
          
          {/* Enhanced Status Badges */}
          <div className="flex flex-wrap gap-3">
            <Badge variant="secondary" className="bg-blue-500/20 text-blue-300 px-4 py-2 text-sm font-bold">
              <Zap className="w-4 h-4 mr-2" />
              PowerBoost Ready
            </Badge>
            <Badge variant="secondary" className="bg-green-500/20 text-green-300 px-4 py-2 text-sm font-bold">
              <Shield className="w-4 h-4 mr-2" />
              Supabase Secured
            </Badge>
            <Badge variant="secondary" className="bg-purple-500/20 text-purple-300 px-4 py-2 text-sm font-bold">
              <Database className="w-4 h-4 mr-2" />
              Real-time Database
            </Badge>
            <Badge variant="secondary" className="bg-orange-500/20 text-orange-300 px-4 py-2 text-sm font-bold">
              <Activity className="w-4 h-4 mr-2" />
              Live Analytics
            </Badge>
            {isAuthenticated && user && (
              <Badge variant="secondary" className="bg-emerald-500/20 text-emerald-300 px-4 py-2 text-sm font-bold">
                <Crown className="w-4 h-4 mr-2" />
                {user.totalUpvotes.toLocaleString()} Total Upvotes
              </Badge>
            )}
          </div>
        </div>

        // Lanjutkan dari bagian yang terpotong, tambahkan semua komponen UI yang hilang

        {/* Optimized Main Content Grid */}
        <div className="grid grid-cols-1 2xl:grid-cols-12 xl:grid-cols-8 lg:grid-cols-6 gap-6">
          {/* Main Control Panel - Optimized Size */}
          <div className="2xl:col-span-6 xl:col-span-5 lg:col-span-4 col-span-1 space-y-6">
            <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
              <TabsList className={`grid w-full grid-cols-2 p-2 rounded-2xl h-16 ${
                isDarkMode ? 'bg-slate-900/50' : 'bg-white/50'
              } backdrop-blur-xl border-2 ${
                isDarkMode ? 'border-slate-700/50' : 'border-white/30'
              }`}>
                <TabsTrigger 
                  value="upvotes" 
                  className="data-[state=active]:bg-gradient-to-r data-[state=active]:from-blue-500 data-[state=active]:to-purple-600 data-[state=active]:text-white rounded-xl py-4 font-bold text-base h-12"
                >
                  <ArrowUp className="w-5 h-5 mr-2" />
                  Server Upvotes
                </TabsTrigger>
                <TabsTrigger 
                  value="powerboost" 
                  className="data-[state=active]:bg-gradient-to-r data-[state=active]:from-purple-500 data-[state=active]:to-pink-600 data-[state=active]:text-white rounded-xl py-4 font-bold text-base h-12"
                >
                  <Power className="w-5 h-5 mr-2" />
                  PowerBoost
                </TabsTrigger>
              </TabsList>

              {/* Enhanced Upvotes Tab */}
              <TabsContent value="upvotes" className="space-y-6 mt-6">
                {/* Server Configuration Card - Optimized */}
                <Card className={`backdrop-blur-xl border-2 shadow-2xl rounded-3xl overflow-hidden ${
                  isDarkMode ? 'bg-slate-900/40 border-slate-700/50' : 'bg-white/60 border-white/30'
                }`}>
                  <CardHeader className="pb-6">
                    <CardTitle className="flex items-center gap-3 text-2xl font-bold">
                      <div className="p-3 bg-gradient-to-r from-blue-500 to-purple-600 rounded-xl">
                        <Server className="w-7 h-7 text-white" />
                      </div>
                      Server Configuration & Validation
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-8">
                    {/* Server Input Section - Better Layout */}
                    <div className="grid grid-cols-1 gap-8">
                      <div className="space-y-4">
                        <label className={`text-base font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                          🌐 Server Address
                        </label>
                        <div className="relative">
                          <Input
                            value={serverInput}
                            onChange={(e) => setServerInput(e.target.value)}
                            placeholder="cfx.re/join/xxx123 or IP:PORT"
                            className={`pl-14 pr-28 py-5 text-lg font-medium rounded-2xl border-2 ${
                              isDarkMode 
                                ? 'bg-slate-800/50 border-slate-600 text-white focus:border-blue-400' 
                                : 'bg-white/50 border-slate-300 text-slate-900 focus:border-blue-500'
                            }`}
                          />
                          <Server className="absolute left-5 top-1/2 transform -translate-y-1/2 w-6 h-6 text-slate-400" />
                          
                          {/* Status Indicators */}
                          <div className="absolute right-5 top-1/2 transform -translate-y-1/2 flex items-center gap-3">
                            {serverStatus === 'validating' && (
                              <div className="w-6 h-6 border-2 border-blue-400 border-t-transparent rounded-full animate-spin" />
                            )}
                            {serverStatus === 'valid' && (
                              <div className="w-6 h-6 bg-green-500 rounded-full animate-pulse shadow-lg shadow-green-500/50" />
                            )}
                            {serverStatus === 'invalid' && (
                              <div className="w-6 h-6 bg-red-500 rounded-full animate-pulse shadow-lg shadow-red-500/50" />
                            )}
                            {serverInput && (
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={copyServerAddress}
                               className="p-2"
                              >
                                <Copy className="w-5 h-5" />
                              </Button>
                            )}
                          </div>
                        </div>
                      </div>
                      
                      <div className="space-y-4">
                        <label className={`text-base font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                          🎯 Upvotes Amount
                        </label>
                        <div className="relative">
                          <Input
                            type="number"
                            value={upvotesInput}
                            onChange={(e) => setUpvotesInput(e.target.value)}
                            placeholder="Enter upvotes amount"
                            min="1"
                            max="50000"
                            className={`pl-14 py-5 text-lg font-medium rounded-2xl border-2 ${
                              isDarkMode 
                                ? 'bg-slate-800/50 border-slate-600 text-white focus:border-purple-400' 
                                : 'bg-white/50 border-slate-300 text-slate-900 focus:border-purple-500'
                            }`}
                          />
                          <Target className="absolute left-5 top-1/2 transform -translate-y-1/2 w-6 h-6 text-slate-400" />
                        </div>
                        <div className="flex items-center justify-between">
                          <p className={`text-sm ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                            💡 Range: 1 - 50,000 upvotes
                          </p>
                          {upvotesInput && parseInt(upvotesInput) > 0 && (
                            <Badge className="bg-blue-500/20 text-blue-300 px-3 py-1">
                              ⏱️ ~{Math.ceil(parseInt(upvotesInput) / 1000)}s
                            </Badge>
                          )}
                        </div>
                      </div>
                    </div>

                    {/* Action Buttons - Better Spacing */}
                    <div className="flex gap-4">
                      <Button
                        onClick={() => validateServer(serverInput)}
                        disabled={!serverInput.trim() || isRefreshing}
                        className="flex-1 bg-gradient-to-r from-emerald-500 to-teal-600 hover:from-emerald-600 hover:to-teal-700 py-4 rounded-2xl font-bold text-base"
                      >
                        {isRefreshing ? (
                          <RefreshCw className="w-5 h-5 mr-2 animate-spin" />
                        ) : (
                          <Search className="w-5 h-5 mr-2" />
                        )}
                        Validate Server
                      </Button>
                      {serverInfo && isAuthenticated && (
                        <Button
                          onClick={addToFavorites}
                          variant="outline"
                          className="px-6 py-4 rounded-2xl"
                        >
                          <Heart className="w-5 h-5" />
                        </Button>
                      )}
                    </div>

                    {/* Enhanced Server Information Display - Optimized */}
                    {serverInfo && (
                      <div className={`p-8 rounded-3xl border-2 ${
                        isDarkMode 
                          ? 'bg-gradient-to-r from-slate-800/50 to-slate-700/50 border-slate-600' 
                          : 'bg-gradient-to-r from-white/50 to-slate-50/50 border-slate-200'
                      }`}>
                        <div className="flex items-start justify-between mb-6">
                          <div className="flex items-center gap-5">
                            <div className="relative">
                              {serverInfo.icon ? (
                                <img 
                                  src={serverInfo.icon || "/placeholder.svg"} 
                                  alt="Server Icon" 
                                  className="w-20 h-20 rounded-2xl object-cover"
                                />
                              ) : (
                                <div className="w-20 h-20 bg-gradient-to-r from-blue-500 to-purple-600 rounded-2xl flex items-center justify-center">
                                  <Server className="w-10 h-10 text-white" />
                                </div>
                              )}
                              {serverInfo.premium && (
                                <div className="absolute -top-2 -right-2 w-8 h-8 bg-gradient-to-r from-yellow-400 to-orange-500 rounded-full flex items-center justify-center">
                                  <Crown className="w-4 h-4 text-white" />
                                </div>
                              )}
                            </div>
                            <div>
                              <h3 className={`text-2xl font-bold mb-2 ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                {serverInfo.hostname}
                              </h3>
                              <div className="flex items-center gap-4">
                                <Badge className="bg-green-500/20 text-green-400 font-bold px-3 py-1">
                                  <Users className="w-4 h-4 mr-1" />
                                  {serverInfo.clients}/{serverInfo.sv_maxclients}
                                </Badge>
                                <Badge className="bg-blue-500/20 text-blue-400 font-bold px-3 py-1">
                                  <MapPin className="w-4 h-4 mr-1" />
                                  {serverInfo.country}
                                </Badge>
                                <Badge className="bg-purple-500/20 text-purple-400 font-bold px-3 py-1">
                                  <Zap className="w-4 h-4 mr-1" />
                                  {serverInfo.upvotePower}
                                </Badge>
                              </div>
                            </div>
                          </div>
                          
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => setShowServerDetails(!showServerDetails)}
                            className="p-3"
                          >
                            {showServerDetails ? <ChevronUp className="w-5 h-5" /> : <ChevronDown className="w-5 h-5" />}
                          </Button>
                        </div>

                        {/* Real-time Server Metrics */}
                        <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                          <div className={`p-4 rounded-2xl ${
                            isDarkMode ? 'bg-slate-700/50' : 'bg-white/50'
                          }`}>
                            <p className={`text-sm font-medium ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                              Ping
                            </p>
                            <p className={`font-bold text-lg ${
                              serverMetrics.ping > 100 ? 'text-red-400' : 
                              serverMetrics.ping > 50 ? 'text-yellow-400' : 'text-green-400'
                            }`}>
                              {serverMetrics.ping}ms
                            </p>
                          </div>
                          <div className={`p-4 rounded-2xl ${
                            isDarkMode ? 'bg-slate-700/50' : 'bg-white/50'
                          }`}>
                            <p className={`text-sm font-medium ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                              Load
                            </p>
                            <p className={`font-bold text-lg ${
                              serverMetrics.serverLoad > 80 ? 'text-red-400' : 
                              serverMetrics.serverLoad > 60 ? 'text-yellow-400' : 'text-green-400'
                            }`}>
                              {serverMetrics.serverLoad}%
                            </p>
                          </div>
                          <div className={`p-4 rounded-2xl ${
                            isDarkMode ? 'bg-slate-700/50' : 'bg-white/50'
                          }`}>
                            <p className={`text-sm font-medium ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                              Memory
                            </p>
                            <p className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                              {serverMetrics.memoryUsage}%
                            </p>
                          </div>
                          <div className={`p-4 rounded-2xl ${
                            isDarkMode ? 'bg-slate-700/50' : 'bg-white/50'
                          }`}>
                            <p className={`text-sm font-medium ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                              CPU
                            </p>
                            <p className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                              {serverMetrics.cpuUsage}%
                            </p>
                          </div>
                        </div>

                        {/* Detailed Server Info (Collapsible) - Better Spacing */}
                        {showServerDetails && (
                          <div className="space-y-6 border-t border-slate-300 dark:border-slate-600 pt-6">
                            {/* Server Features */}
                            <div>
                              <h4 className={`font-bold text-lg mb-3 ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                🚀 Server Features
                              </h4>
                              <div className="flex flex-wrap gap-3">
                                {serverInfo.onesync && (
                                  <Badge className="bg-emerald-500/20 text-emerald-400 px-3 py-1">
                                    <RefreshCw className="w-4 h-4 mr-1" />
                                    OneSync
                                  </Badge>
                                )}
                                {serverInfo.premium && (
                                  <Badge className="bg-yellow-500/20 text-yellow-400 px-3 py-1">
                                    <Crown className="w-4 h-4 mr-1" />
                                    Premium
                                  </Badge>
                                )}
                                {serverInfo.burst && (
                                  <Badge className="bg-red-500/20 text-red-400 px-3 py-1">
                                    <Zap className="w-4 h-4 mr-1" />
                                    Burst
                                  </Badge>
                                )}
                                <Badge className="bg-blue-500/20 text-blue-400 px-3 py-1">
                                  <Shield className="w-4 h-4 mr-1" />
                                  Pure Level {serverInfo.pureLevel}
                                </Badge>
                              </div>
                            </div>

                            {/* Server Tags */}
                            {serverInfo.tags.length > 0 && (
                              <div>
                                <h4 className={`font-bold text-lg mb-3 ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                  🏷️ Tags
                                </h4>
                                <div className="flex flex-wrap gap-2">
                                  {serverInfo.tags.slice(0, 8).map((tag, index) => (
                                    <Badge key={index} variant="outline" className="text-sm px-3 py-1">
                                      {tag}
                                    </Badge>
                                  ))}
                                  {serverInfo.tags.length > 8 && (
                                    <Badge variant="outline" className="text-sm px-3 py-1">
                                      +{serverInfo.tags.length - 8} more
                                    </Badge>
                                  )}
                                </div>
                              </div>
                            )}

                            {/* Resources */}
                            {serverInfo.resources.length > 0 && (
                              <div>
                                <h4 className={`font-bold text-lg mb-3 ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                  🔧 Resources ({serverInfo.resources.length})
                                </h4>
                                <div className={`p-4 rounded-2xl ${
                                  isDarkMode ? 'bg-slate-800/50' : 'bg-slate-100'
                                }`}>
                                  <p className={`text-sm ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                                    {serverInfo.resources.slice(0, 10).join(', ')}
                                    {serverInfo.resources.length > 10 && ` and ${serverInfo.resources.length - 10} more...`}
                                  </p>
                                </div>
                              </div>
                            )}
                          </div>
                        )}
                      </div>
                    )}

                    {/* Progress Section - Better Layout */}
                    <div className="space-y-6">
                      <div className="flex justify-between items-center">
                        <span className={`text-xl font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                          🚀 Process Progress
                        </span>
                        <span className="text-3xl font-bold text-transparent bg-gradient-to-r from-blue-400 to-purple-400 bg-clip-text">
                          {Math.floor(state.progress)}%
                        </span>
                      </div>
                      <div className="relative">
                        <Progress 
                          value={state.progress} 
                          className="h-6 rounded-full overflow-hidden"
                        />
                        <div className="absolute inset-0 bg-gradient-to-r from-blue-500/20 to-purple-500/20 rounded-full animate-pulse" />
                      </div>
                      
                      {/* Real-time Stats - Better Grid */}
                      {state.isRunning && (
                        <div className="grid grid-cols-3 gap-4">
                          <div className={`p-5 rounded-2xl text-center ${
                            isDarkMode ? 'bg-blue-500/10 border border-blue-500/20' : 'bg-blue-100 border border-blue-200'
                          }`}>
                            <p className={`text-sm font-medium ${isDarkMode ? 'text-blue-300' : 'text-blue-700'}`}>
                              Current Upvotes
                            </p>
                            <p className="text-2xl font-bold text-blue-400">
                              {state.totalUpvotes.toLocaleString()}
                            </p>
                          </div>
                          <div className={`p-5 rounded-2xl text-center ${
                            isDarkMode ? 'bg-green-500/10 border border-green-500/20' : 'bg-green-100 border border-green-200'
                          }`}>
                            <p className={`text-sm font-medium ${isDarkMode ? 'text-green-300' : 'text-green-700'}`}>
                              Speed
                            </p>
                            <p className="text-2xl font-bold text-green-400">
                              {getCurrentSpeed()}/s
                            </p>
                          </div>
                          <div className={`p-5 rounded-2xl text-center ${
                            isDarkMode ? 'bg-purple-500/10 border border-purple-500/20' : 'bg-purple-100 border border-purple-200'
                          }`}>
                            <p className={`text-sm font-medium ${isDarkMode ? 'text-purple-300' : 'text-purple-700'}`}>
                              Time Elapsed
                            </p>
                            <p className="text-2xl font-bold text-purple-400">
                              {formatElapsedTime()}
                            </p>
                          </div>
                        </div>
                      )}
                    </div>

                    {/* Action Buttons - Better Sizing */}
                    <div className="flex gap-6">
                      <Button
                        onClick={startProcess}
                        disabled={state.isRunning || !serverInfo}
                        className="flex-1 bg-gradient-to-r from-blue-500 to-purple-600 hover:from-blue-600 hover:to-purple-700 py-5 rounded-2xl font-bold text-lg shadow-2xl shadow-blue-500/25"
                      >
                        <Play className="w-6 h-6 mr-2" />
                        {state.isRunning ? 'Processing...' : 'Start Upvotes'}
                      </Button>
                      <Button
                        onClick={stopProcess}
                        disabled={!state.isRunning}
                        variant="destructive"
                        className="flex-1 py-5 rounded-2xl font-bold text-lg shadow-2xl shadow-red-500/25"
                      >
                        <Square className="w-6 h-6 mr-2" />
                        Stop Process
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Enhanced PowerBoost Tab - Better Layout */}
              <TabsContent value="powerboost" className="space-y-6 mt-6">
                <Card className={`backdrop-blur-xl border-2 shadow-2xl rounded-3xl overflow-hidden ${
                  isDarkMode ? 'bg-slate-900/40 border-slate-700/50' : 'bg-white/60 border-white/30'
                }`}>
                  <CardHeader className="pb-6">
                    <CardTitle className="flex items-center gap-3 text-2xl font-bold">
                      <div className="p-3 bg-gradient-to-r from-purple-500 to-pink-600 rounded-xl">
                        <Power className="w-7 h-7 text-white" />
                      </div>
                      PowerBoost Enhancement System
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-8">
                    <div className="grid grid-cols-1 gap-8">
                      <div className="space-y-4">
                        <label className={`text-base font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                          ⚡ Boost Type
                        </label>
                        <select 
                          value={powerBoostType}
                          onChange={(e) => setPowerBoostType(e.target.value)}
                          className={`w-full rounded-2xl px-5 py-5 text-lg font-medium border-2 ${
                            isDarkMode 
                              ? 'bg-slate-800/50 border-slate-600 text-white focus:border-purple-400' 
                              : 'bg-white/50 border-slate-300 text-slate-900 focus:border-purple-500'
                          }`}
                        >
                          <option value="performance">🚀 Performance Boost (+200% Speed)</option>
                          <option value="visibility">👁️ Visibility Boost (+300% Exposure)</option>
                          <option value="ranking">📈 Ranking Boost (Top 10 Priority)</option>
                          <option value="premium">👑 Premium Boost (All Features)</option>
                        </select>
                      </div>
                      <div className="space-y-4">
                        <label className={`text-base font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                          ⏰ Duration (Hours)
                        </label>
                        <div className="relative">
                          <Input
                            type="number"
                            value={powerBoostDuration}
                            onChange={(e) => setPowerBoostDuration(e.target.value)}
                            placeholder="1-168 hours"
                            min="1"
                            max="168"
                            className={`pl-14 py-5 text-lg font-medium rounded-2xl border-2 ${
                              isDarkMode 
                                ? 'bg-slate-800/50 border-slate-600 text-white focus:border-pink-400' 
                                : 'bg-white/50 border-slate-300 text-slate-900 focus:border-pink-500'
                            }`}
                          />
                          <Timer className="absolute left-5 top-1/2 transform -translate-y-1/2 w-6 h-6 text-slate-400" />
                        </div>
                      </div>
                    </div>
                    
                    {/* PowerBoost Features - Better Layout */}
                    <div className={`rounded-3xl p-8 border-2 ${
                      isDarkMode 
                        ? 'bg-gradient-to-r from-purple-500/10 to-pink-500/10 border-purple-500/20' 
                        : 'bg-gradient-to-r from-purple-100 to-pink-100 border-purple-200'
                    }`}>
                      <h4 className={`font-bold text-2xl mb-6 ${
                        isDarkMode ? 'text-purple-300' : 'text-purple-700'
                      }`}>
                        🌟 PowerBoost Features & Benefits
                      </h4>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div className="space-y-4">
                          <div className="flex items-center gap-4">
                            <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-cyan-500 rounded-full flex items-center justify-center">
                              <Eye className="w-6 h-6 text-white" />
                            </div>
                            <div>
                              <p className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                Enhanced Visibility
                              </p>
                              <p className={`text-sm ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                                +300% server exposure
                              </p>
                            </div>
                          </div>
                          <div className="flex items-center gap-4">
                            <div className="w-12 h-12 bg-gradient-to-r from-green-500 to-emerald-500 rounded-full flex items-center justify-center">
                              <TrendingUp className="w-6 h-6 text-white" />
                            </div>
                            <div>
                              <p className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                Priority Ranking
                              </p>
                              <p className={`text-sm ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                                Top 10 guaranteed placement
                              </p>
                            </div>
                          </div>
                          <div className="flex items-center gap-4">
                            <div className="w-12 h-12 bg-gradient-to-r from-purple-500 to-pink-500 rounded-full flex items-center justify-center">
                              <BarChart3 className="w-6 h-6 text-white" />
                            </div>
                            <div>
                              <p className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                Advanced Analytics
                              </p>
                              <p className={`text-sm ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                                Real-time performance data
                              </p>
                            </div>
                          </div>
                        </div>
                        <div className="space-y-4">
                          <div className="flex items-center gap-4">
                            <div className="w-12 h-12 bg-gradient-to-r from-orange-500 to-red-500 rounded-full flex items-center justify-center">
                              <Zap className="w-6 h-6 text-white" />
                            </div>
                            <div>
                              <p className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                Speed Multiplier
                              </p>
                              <p className={`text-sm ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                                5x faster upvote processing
                              </p>
                            </div>
                          </div>
                          <div className="flex items-center gap-4">
                            <div className="w-12 h-12 bg-gradient-to-r from-yellow-500 to-orange-500 rounded-full flex items-center justify-center">
                              <Crown className="w-6 h-6 text-white" />
                            </div>
                            <div>
                              <p className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                Premium Badges
                              </p>
                              <p className={`text-sm ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                                Custom server verification
                              </p>
                            </div>
                          </div>
                          <div className="flex items-center gap-4">
                            <div className="w-12 h-12 bg-gradient-to-r from-indigo-500 to-purple-500 rounded-full flex items-center justify-center">
                              <Shield className="w-6 h-6 text-white" />
                            </div>
                            <div>
                              <p className={`font-bold text-lg ${isDarkMode ? 'text-white' : 'text-slate-900'}`}>
                                DDoS Protection
                              </p>
                              <p className={`text-sm ${isDarkMode ? 'text-slate-400' : 'text-slate-600'}`}>
                                Enhanced security layer
                              </p>
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                    
                    {/* Pricing Information - Better Design */}
                    {powerBoostDuration && parseInt(powerBoostDuration) > 0 && (
                      <div className={`p-6 rounded-2xl ${
                        isDarkMode ? 'bg-emerald-500/10 border border-emerald-500/20' : 'bg-emerald-100 border border-emerald-200'
                      }`}>
                        <div className="flex items-center justify-between">
                          <div>
                            <p className={`font-bold text-lg ${isDarkMode ? 'text-emerald-300' : 'text-emerald-700'}`}>
                              💰 Estimated Cost
                            </p>
                            <p className={`text-sm ${isDarkMode ? 'text-emerald-400' : 'text-emerald-600'}`}>
                              {powerBoostType.charAt(0).toUpperCase() + powerBoostType.slice(1)} boost for {powerBoostDuration} hours
                            </p>
                          </div>
                          <div className="text-right">
                            <p className="text-3xl font-bold text-emerald-400">
                              ${(parseInt(powerBoostDuration) * (powerBoostType === 'premium' ? 5 : powerBoostType === 'ranking' ? 3 : 2)).toFixed(2)}
                            </p>
                            <p className={`text-sm ${isDarkMode ? 'text-emerald-400' : 'text-emerald-600'}`}>
                              ${powerBoostType === 'premium' ? '5' : powerBoostType === 'ranking' ? '3' : '2'}/hour
                            </p>
                          </div>
                        </div>
                      </div>
                    )}
                    
                    <Button 
                      onClick={activatePowerBoost}
                      disabled={!serverInfo || !powerBoostDuration}
                      className="w-full bg-gradient-to-r from-purple-500 to-pink-600 hover:from-purple-600 hover:to-pink-700 py-5 rounded-2xl font-bold text-lg shadow-2xl shadow-purple-500/25"
                    >
                      <Zap className="w-6 h-6 mr-2" />
                      Activate PowerBoost
                    </Button>
                  </CardContent>
                </Card>
              </TabsContent>
            </Tabs>
          </div>

          {/* Resizable FiveM Browser Panel */}
          <div className="2xl:col-span-3 xl:col-span-3 lg:col-span-2 col-span-1">
            <Card className={`backdrop-blur-xl border-2 shadow-2xl rounded-3xl overflow-hidden h-fit ${
              isDarkMode ? 'bg-slate-900/40 border-slate-700/50' : 'bg-white/60 border-white/30'
            } ${isBrowserMaximized ? 'fixed inset-4 z-50 h-auto' : ''}`}>
              <CardHeader className="pb-4">
                <CardTitle className="flex items-center gap-3 text-xl font-bold">
                  <div className="p-2 bg-gradient-to-r from-orange-500 to-red-600 rounded-xl">
                    <Globe className="w-6 h-6 text-white" />
                  </div>
                  FiveM Servers Browser
                  <div className="ml-auto flex items-center gap-2">
                    <Badge className="bg-green-500/20 text-green-400">
                      <Wifi className="w-3 h-3 mr-1" />
                      Live
                    </Badge>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={toggleBrowserMaximize}
                      title={isBrowserMaximized ? "Minimize" : "Maximize"}
                    >
                      {isBrowserMaximized ? <Minimize2 className="w-4 h-4" /> : <Maximize2 className="w-4 h-4" />}
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => window.open('https://servers.fivem.net/', '_blank')}
                    >
                      <ExternalLink className="w-4 h-4" />
                    </Button>
                  </div>
                </CardTitle>
              </CardHeader>
              <CardContent className="p-0">
                <div 
                  className="rounded-2xl overflow-hidden border-2 border-slate-200 dark:border-slate-700 resize"
                  style={{ 
                    width: isBrowserMaximized ? '100%' : `${browserPanelSize.width}px`, 
                    height: `${browserPanelSize.height}px`,
                    maxWidth: '100%',
                    minWidth: '400px',
                    minHeight: '300px'
                  }}
                >
                  <iframe
                    src="https://servers.fivem.net/"
                    className="w-full h-full border-0"
                    title="FiveM Servers Browser"
                    sandbox="allow-same-origin allow-scripts allow-forms allow-popups allow-top-navigation"
                  />
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Stats & Analytics Sidebar - Optimized Size */}
          <div className="2xl:col-span-3 xl:col-span-8 lg:col-span-6 col-span-1 space-y-6">
            {/* Real-time Stats Cards - Better Grid */}
            <div className="grid grid-cols-1 xl:grid-cols-3 2xl:grid-cols-1 gap-4">
              <Card className="bg-gradient-to-br from-blue-500/20 to-purple-600/20 backdrop-blur-xl border-2 border-blue-500/20 rounded-2xl shadow-2xl">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-600'}`}>
                        🎯 Current Session
                      </p>
                      <p className="text-3xl font-black text-transparent bg-gradient-to-r from-blue-400 to-purple-400 bg-clip-text">
                        {state.totalUpvotes.toLocaleString()}
                      </p>
                      <p className={`text-xs ${isDarkMode ? 'text-slate-400' : 'text-slate-500'}`}>
                        Target: {state.targetUpvotes.toLocaleString()}
                      </p>
                    </div>
                    <div className="relative">
                      <div className="w-16 h-16 bg-gradient-to-r from-blue-500 to-purple-600 rounded-2xl flex items-center justify-center shadow-lg shadow-blue-500/25">
                        <ArrowUp className="w-8 h-8 text-white" />
                      </div>
                      {state.isRunning && (
                        <div className="absolute inset-0 bg-gradient-to-r from-blue-500 to-purple-600 rounded-2xl animate-pulse opacity-50" />
                      )}
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-green-500/20 to-emerald-600/20 backdrop-blur-xl border-2 border-green-500/20 rounded-2xl shadow-2xl">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-600'}`}>
                        ⚡ Processing Speed
                      </p>
                      <p className="text-3xl font-black text-transparent bg-gradient-to-r from-green-400 to-emerald-400 bg-clip-text">
                        {getCurrentSpeed().toLocaleString()}/s
                      </p>
                      <p className={`text-xs ${isDarkMode ? 'text-slate-400' : 'text-slate-500'}`}>
                        Upvotes per second
                      </p>
                    </div>
                    <div className="w-16 h-16 bg-gradient-to-r from-green-500 to-emerald-600 rounded-2xl flex items-center justify-center shadow-lg shadow-green-500/25">
                      <Zap className="w-8 h-8 text-white" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-purple-500/20 to-pink-600/20 backdrop-blur-xl border-2 border-purple-500/20 rounded-2xl shadow-2xl">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-600'}`}>
                        ⏱️ Session Time
                      </p>
                      <p className="text-3xl font-black text-transparent bg-gradient-to-r from-purple-400 to-pink-400 bg-clip-text">
                        {formatElapsedTime()}
                      </p>
                      <p className={`text-xs ${isDarkMode ? 'text-slate-400' : 'text-slate-500'}`}>
                        Minutes:Seconds
                      </p>
                    </div>
                    <div className="w-16 h-16 bg-gradient-to-r from-purple-500 to-pink-600 rounded-2xl flex items-center justify-center shadow-lg shadow-purple-500/25">
                      <Clock className="w-8 h-8 text-white" />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Real-time Analytics Chart - Better Size */}
            {realtimeAnalytics.length > 0 && (
              <Card className={`backdrop-blur-xl border-2 shadow-2xl rounded-2xl ${
                isDarkMode ? 'bg-slate-900/40 border-slate-700/50' : 'bg-white/60 border-white/30'
              }`}>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-lg font-bold">
                    <Activity className="w-5 h-5 text-emerald-400" />
                    Real-time Analytics
                    <Badge className="bg-emerald-500/20 text-emerald-400 ml-auto">
                      <Database className="w-3 h-3 mr-1" />
                      Live Data
                    </Badge>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="h-[300px]">
                    <ResponsiveContainer width="100%" height="100%">
                      <AreaChart data={realtimeAnalytics}>
                        <defs>
                          <linearGradient id="upvotesGradient" x1="0" y1="0" x2="0" y2="1">
                            <stop offset="5%" stopColor="#3b82f6" stopOpacity={0.8}/>
                            <stop offset="95%" stopColor="#3b82f6" stopOpacity={0.1}/>
                          </linearGradient>
                        </defs>
                        <CartesianGrid strokeDasharray="3 3" stroke={isDarkMode ? '#374151' : '#e5e7eb'} />
                        <XAxis 
                          dataKey="time" 
                          stroke={isDarkMode ? '#9ca3af' : '#6b7280'}
                          fontSize={12}
                        />
                        <YAxis 
                          stroke={isDarkMode ? '#9ca3af' : '#6b7280'}
                          fontSize={12}
                        />
                        <Tooltip 
                          contentStyle={{
                            backgroundColor: isDarkMode ? '#1f2937' : '#ffffff',
                            border: `1px solid ${isDarkMode ? '#374151' : '#e5e7eb'}`,
                            borderRadius: '12px',
                            boxShadow: '0 10px 25px rgba(0,0,0,0.1)'
                          }}
                        />
                        <Area
                          type="monotone"
                          dataKey="upvotes"
                          stroke="#3b82f6"
                          fillOpacity={1}
                          fill="url(#upvotesGradient)"
                          strokeWidth={3}
                        />
                      </AreaChart>
                    </ResponsiveContainer>
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Favorite Servers - Better Layout */}
            {isAuthenticated && favoriteServers.length > 0 && (
              <Card className={`backdrop-blur-xl border-2 shadow-2xl rounded-2xl ${
                isDarkMode ? 'bg-slate-900/40 border-slate-700/50' : 'bg-white/60 border-white/30'
              }`}>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-lg font-bold">
                    <Heart className="w-5 h-5 text-red-400" />
                    Favorite Servers
                    <Badge className="bg-red-500/20 text-red-400 ml-auto">
                      {favoriteServers.length}
                    </Badge>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <ScrollArea className="h-[250px]">
                    <div className="space-y-3">
                      {favoriteServers.map((server) => (
                        <div
                          key={server.id}
                          className={`p-4 rounded-xl border-2 transition-all hover:scale-[1.02] ${
                            isDarkMode ? 'bg-slate-800/50 border-slate-600 hover:border-slate-500' : 'bg-white/50 border-slate-200 hover:border-slate-300'
                          }`}
                        >
                          <div className="flex items-center justify-between">
                            <div className="flex-1 min-w-0">
                              <div className="flex items-center gap-2 mb-1">
                                <p className={`font-bold truncate ${
                                  isDarkMode ? 'text-white' : 'text-slate-900'
                                }`}>
                                  {server.name}
                                </p>
                                <div className={`w-2 h-2 rounded-full ${
                                  server.status === 'online' ? 'bg-green-400' : 'bg-red-400'
                                } animate-pulse`} />
                              </div>
                              <div className="flex items-center gap-4">
                                <p className={`text-sm ${
                                  isDarkMode ? 'text-slate-400' : 'text-slate-600'
                                }`}>
                                  👥 {server.players}/{server.maxPlayers}
                                </p>
                                <p className={`text-sm ${
                                  isDarkMode ? 'text-slate-400' : 'text-slate-600'
                                }`}>
                                  📡 {server.ping}ms
                                </p>
                              </div>
                            </div>
                            <div className="flex items-center gap-2">
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={() => {
                                  setServerInput(server.address)
                                  validateServer(server.address)
                                }}
                              >
                                <RefreshCw className="w-4 h-4" />
                              </Button>
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={() => removeFromFavorites(server.id)}
                              >
                                <Trash2 className="w-4 h-4 text-red-400" />
                              </Button>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  </ScrollArea>
                </CardContent>
              </Card>
            )}

            {/* Resizable Real-time Synchronized Console */}
            <Card className={`backdrop-blur-xl border-2 shadow-2xl rounded-2xl ${
              isDarkMode ? 'bg-slate-900/40 border-slate-700/50' : 'bg-white/60 border-white/30'
            } ${isConsoleMaximized ? 'fixed inset-4 z-50' : ''}`}>
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-lg font-bold">
                  <Terminal className="w-5 h-5 text-emerald-400" />
                  Real-time System Console
                  <div className="ml-auto flex items-center gap-2">
                    <Badge className={`${
                      isConsolePaused ? 'bg-yellow-500/20 text-yellow-400' : 'bg-green-500/20 text-green-400'
                    }`}>
                      {isConsolePaused ? <Pause className="w-3 h-3 mr-1" /> : <Activity className="w-3 h-3 mr-1" />}
                      {isConsolePaused ? 'Paused' : 'Live'}
                    </Badge>
                    <Badge className="bg-slate-500/20 text-slate-400">
                      {filteredMessages.length} messages
                    </Badge>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={toggleConsoleMaximize}
                      title={isConsoleMaximized ? "Minimize" : "Maximize"}
                    >
                      {isConsoleMaximized ? <Minimize2 className="w-4 h-4" /> : <Maximize2 className="w-4 h-4" />}
                    </Button>
                  </div>
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {/* Console Controls */}
                <div className="flex items-center gap-2 flex-wrap">
                  <div className="flex items-center gap-2">
                    <Search className="w-4 h-4 text-slate-400" />
                    <Input
                      placeholder="Search console..."
                      value={consoleSearch}
                      onChange={(e) => setConsoleSearch(e.target.value)}
                      className={`w-40 h-8 text-sm ${
                        isDarkMode ? 'bg-slate-800/50 border-slate-600' : 'bg-white/50 border-slate-300'
                      }`}
                    />
                  </div>
                  <select
                    value={consoleFilter}
                    onChange={(e) => setConsoleFilter(e.target.value)}
                    className={`h-8 px-3 text-sm rounded-lg border ${
                      isDarkMode ? 'bg-slate-800/50 border-slate-600 text-white' : 'bg-white/50 border-slate-300 text-slate-900'
                    }`}
                  >
                    <option value="all">All Messages</option>
                    <option value="info">Info</option>
                    <option value="success">Success</option>
                    <option value="warning">Warning</option>
                    <option value="error">Error</option>
                    <option value="PROCESS">Process</option>
                    <option value="NETWORK">Network</option>
                    <option value="SECURITY">Security</option>
                    <option value="SYSTEM">System</option>
                    <option value="DATABASE">Database</option>
                  </select>
                  <div className="flex items-center gap-1 ml-auto">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={toggleConsolePause}
                      className="h-8 px-2"
                    >
                      {isConsolePaused ? <Play className="w-4 h-4" /> : <Pause className="w-4 h-4" />}
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={clearConsole}
                      className="h-8 px-2"
                    >
                      <X className="w-4 h-4" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={exportConsoleLog}
                      className="h-8 px-2"
                    >
                      <Copy className="w-4 h-4" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => setConsoleAutoScroll(!consoleAutoScroll)}
                      className={`h-8 px-2 ${consoleAutoScroll ? 'text-green-400' : 'text-slate-400'}`}
                    >
                      <ChevronDown className="w-4 h-4" />
                    </Button>
                  </div>
                </div>

                {/* Console Messages */}
                <ScrollArea 
                  className="w-full resize-y" 
                  ref={consoleRef}
                  style={{ 
                    height: isConsoleMaximized ? 'calc(100vh - 300px)' : `${consolePanelSize.height}px`,
                    minHeight: '200px',
                    maxHeight: isConsoleMaximized ? 'calc(100vh - 300px)' : '800px'
                  }}
                >
                  <div className="space-y-1 font-mono text-sm">
                    {filteredMessages.map((msg) => {
                      const getIcon = (type: string) => {
                        switch (type) {
                          case 'success': return <CheckCircle className="w-4 h-4" />
                          case 'error': return <XCircle className="w-4 h-4" />
                          case 'warning': return <AlertCircle className="w-4 h-4" />
                          case 'info': return <Info className="w-4 h-4" />
                          default: return <Terminal className="w-4 h-4" />
                        }
                      }

                      return (
                        <div
                          key={msg.id}
                          className={`flex gap-3 p-3 rounded-lg transition-all hover:scale-[1.01] ${
                            msg.type === 'success' ? 'bg-green-500/10 text-green-400 border-l-2 border-green-500' :
                            msg.type === 'error' ? 'bg-red-500/10 text-red-400 border-l-2 border-red-500' :
                            msg.type === 'warning' ? 'bg-yellow-500/10 text-yellow-400 border-l-2 border-yellow-500' :
                            msg.type === 'system' ? 'bg-purple-500/10 text-purple-400 border-l-2 border-purple-500' :
                            msg.type === 'network' ? 'bg-cyan-500/10 text-cyan-400 border-l-2 border-cyan-500' :
                            msg.type === 'process' ? 'bg-orange-500/10 text-orange-400 border-l-2 border-orange-500' :
                            'bg-blue-500/10 text-blue-400 border-l-2 border-blue-500'
                          }`}
                        >
                          <div className="flex items-center gap-2 min-w-0">
                            {getIcon(msg.type)}
                            <span className={`text-xs ${isDarkMode ? 'text-slate-500' : 'text-slate-400'} min-w-[80px]`}>
                              {msg.timestamp}
                            </span>
                            <Badge variant="outline" className="text-xs px-2 py-0 h-5">
                              {msg.category}
                            </Badge>
                          </div>
                          <span className="flex-1 break-words">{msg.message}</span>
                          {msg.details && (
                            <Button
                              variant="ghost"
                              size="sm"
                              className="h-6 w-6 p-0 opacity-50 hover:opacity-100"
                              onClick={() => console.log('Details:', msg.details)}
                            >
                              <Info className="w-3 h-3" />
                            </Button>
                          )}
                        </div>
                      )
                    })}
                    {filteredMessages.length === 0 && (
                      <div className="text-center py-8 text-slate-400">
                        <Terminal className="w-8 h-8 mx-auto mb-2 opacity-50" />
                        <p>No console messages match your filter</p>
                      </div>
                    )}
                  </div>
                </ScrollArea>
              </CardContent>
            </Card>

            {/* System Status - Better Layout */}
            <Card className={`backdrop-blur-xl border-2 shadow-2xl rounded-2xl ${
              isDarkMode ? 'bg-slate-900/40 border-slate-700/50' : 'bg-white/60 border-white/30'
            }`}>
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-lg font-bold">
                  <Settings className="w-5 h-5 text-slate-400" />
                  System Status
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 gap-3">
                  <div className="flex justify-between items-center p-4 rounded-xl bg-slate-100 dark:bg-slate-800">
                    <div className="flex items-center gap-3">
                      <Users className="w-5 h-5" />
                      <span className={`font-medium ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                        Authentication
                      </span>
                    </div>
                    <Badge variant="secondary" className={
                      isAuthenticated 
                        ? 'bg-green-500/20 text-green-300' 
                        : 'bg-red-500/20 text-red-300'
                    }>
                      {isAuthenticated ? '✅ Authenticated' : '❌ Guest Mode'}
                    </Badge>
                  </div>
                  
                  <div className="flex justify-between items-center p-4 rounded-xl bg-slate-100 dark:bg-slate-800">
                    <div className="flex items-center gap-3">
                      <Database className="w-5 h-5" />
                      <span className={`font-medium ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                        Supabase Database
                      </span>
                    </div>
                    <Badge variant="secondary" className="bg-green-500/20 text-green-300">
                      ✅ Connected
                    </Badge>
                  </div>
                  
                  <div className="flex justify-between items-center p-4 rounded-xl bg-slate-100 dark:bg-slate-800">
                    <div className="flex items-center gap-3">
                      <Server className="w-5 h-5" />
                      <span className={`font-medium ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                        Server Status
                      </span>
                    </div>
                    <Badge 
                      variant="secondary" 
                      className={
                        serverStatus === 'valid' ? 'bg-green-500/20 text-green-300' :
                        serverStatus === 'invalid' ? 'bg-red-500/20 text-red-300' :
                        serverStatus === 'validating' ? 'bg-yellow-500/20 text-yellow-300' :
                        'bg-slate-500/20 text-slate-300'
                      }
                    >
                      {serverStatus === 'valid' ? '✅ Online' :
                       serverStatus === 'invalid' ? '❌ Offline' :
                       serverStatus === 'validating' ? '🔄 Checking' :
                       '⏸️ Idle'}
                    </Badge>
                  </div>
                  
                  <div className="flex justify-between items-center p-4 rounded-xl bg-slate-100 dark:bg-slate-800">
                    <div className="flex items-center gap-3">
                      <Activity className="w-5 h-5" />
                      <span className={`font-medium ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                        Process Status
                      </span>
                    </div>
                    <Badge 
                      variant="secondary" 
                      className={state.isRunning ? 'bg-blue-500/20 text-blue-300' : 'bg-slate-500/20 text-slate-300'}
                    >
                      {state.isRunning ? '🚀 Running' : '⏸️ Idle'}
                    </Badge>
                  </div>
                  
                  <div className="flex justify-between items-center p-4 rounded-xl bg-slate-100 dark:bg-slate-800">
                    <div className="flex items-center gap-3">
                      <Terminal className="w-5 h-5" />
                      <span className={`font-medium ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                        Console Sync
                      </span>
                    </div>
                    <Badge 
                      variant="secondary" 
                      className={!isConsolePaused ? 'bg-green-500/20 text-green-300' : 'bg-yellow-500/20 text-yellow-300'}
                    >
                      {!isConsolePaused ? '🔄 Real-time' : '⏸️ Paused'}
                    </Badge>
                  </div>
                  
                  {isAuthenticated && user && (
                    <div className="flex justify-between items-center p-4 rounded-xl bg-slate-100 dark:bg-slate-800">
                      <div className="flex items-center gap-3">
                        <Crown className="w-5 h-5" />
                        <span className={`font-medium ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                          Account Plan
                        </span>
                      </div>
                      <Badge 
                        variant="secondary" 
                        className={
                          user.plan === 'enterprise' ? 'bg-purple-500/20 text-purple-300' :
                          user.plan === 'premium' ? 'bg-blue-500/20 text-blue-300' :
                          'bg-slate-500/20 text-slate-300'
                        }
                      >
                        👑 {user.plan.toUpperCase()}
                      </Badge>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Authentication Modal */}
        <Dialog open={showAuthModal} onOpenChange={setShowAuthModal}>
          <DialogContent className={`max-w-md ${
            isDarkMode ? 'bg-slate-900 border-slate-700' : 'bg-white border-slate-200'
          } rounded-2xl`}>
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2 text-xl font-bold">
                <User className="w-6 h-6 text-blue-400" />
                {authMode === 'login' ? 'Welcome Back!' : 'Join FiveM UDG'}
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-6">
              <Tabs value={authMode} onValueChange={(value) => setAuthMode(value as 'login' | 'register')}>
                <TabsList className="grid w-full grid-cols-2 rounded-xl">
                  <TabsTrigger value="login" className="rounded-lg">Login</TabsTrigger>
                  <TabsTrigger value="register" className="rounded-lg">Register</TabsTrigger>
                </TabsList>
                
                <TabsContent value="login" className="space-y-4 mt-6">
                  <div className="space-y-2">
                    <label className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                      📧 Email Address
                    </label>
                    <Input
                      type="email"
                      placeholder="Enter your email"
                      className={`py-3 rounded-xl ${
                        isDarkMode 
                          ? 'bg-slate-800/50 border-slate-600 text-white' 
                          : 'bg-slate-50 border-slate-300 text-slate-900'
                      }`}
                      id="login-email"
                    />
                  </div>
                  <div className="space-y-2">
                    <label className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                      🔒 Password
                    </label>
                    <Input
                      type="password"
                      placeholder="Enter your password"
                      className={`py-3 rounded-xl ${
                        isDarkMode 
                          ? 'bg-slate-800/50 border-slate-600 text-white' 
                          : 'bg-slate-50 border-slate-300 text-slate-900'
                      }`}
                      id="login-password"
                    />
                  </div>
                  <div className={`p-4 rounded-xl ${
                    isDarkMode ? 'bg-blue-500/10 border border-blue-500/20' : 'bg-blue-100 border border-blue-200'
                  }`}>
                    <p className={`text-sm font-medium ${isDarkMode ? 'text-blue-300' : 'text-blue-700'}`}>
                      🗄️ Supabase Authentication
                    </p>
                    <p className={`text-xs ${isDarkMode ? 'text-blue-400' : 'text-blue-600'}`}>
                      Secure login with real-time database synchronization.
                    </p>
                  </div>
                  <Button
                    onClick={() => {
                      const email = (document.getElementById('login-email') as HTMLInputElement)?.value
                      const password = (document.getElementById('login-password') as HTMLInputElement)?.value
                      if (email && password) handleLogin(email, password)
                    }}
                    className="w-full bg-gradient-to-r from-blue-500 to-purple-600 hover:from-blue-600 hover:to-purple-700 py-3 rounded-xl font-bold"
                  >
                    🚀 Login to Account
                  </Button>
                </TabsContent>
                
                <TabsContent value="register" className="space-y-4 mt-6">
                  <div className="space-y-2">
                    <label className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                      👤 Username
                    </label>
                    <Input
                      placeholder="Choose a username"
                      className={`py-3 rounded-xl ${
                        isDarkMode 
                          ? 'bg-slate-800/50 border-slate-600 text-white' 
                          : 'bg-slate-50 border-slate-300 text-slate-900'
                      }`}
                      id="register-username"
                    />
                  </div>
                  <div className="space-y-2">
                    <label className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                      📧 Email Address
                    </label>
                    <Input
                      type="email"
                      placeholder="Enter your email"
                      className={`py-3 rounded-xl ${
                        isDarkMode 
                          ? 'bg-slate-800/50 border-slate-600 text-white' 
                          : 'bg-slate-50 border-slate-300 text-slate-900'
                      }`}
                      id="register-email"
                    />
                  </div>
                  <div className="space-y-2">
                    <label className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                      🔒 Password
                    </label>
                    <Input
                      type="password"
                      placeholder="Create a password"
                      className={`py-3 rounded-xl ${
                        isDarkMode 
                          ? 'bg-slate-800/50 border-slate-600 text-white' 
                          : 'bg-slate-50 border-slate-300 text-slate-900'
                      }`}
                      id="register-password"
                    />
                  </div>
                  <div className={`p-4 rounded-xl ${
                    isDarkMode ? 'bg-green-500/10 border border-green-500/20' : 'bg-green-100 border border-green-200'
                  }`}>
                    <p className={`text-sm font-medium ${isDarkMode ? 'text-green-300' : 'text-green-700'}`}>
                      🎉 Create Your Account
                    </p>
                    <p className={`text-xs ${isDarkMode ? 'text-green-400' : 'text-green-600'}`}>
                      Join thousands of users with secure Supabase authentication.
                    </p>
                  </div>
                  <Button
                    onClick={() => {
                      const username = (document.getElementById('register-username') as HTMLInputElement)?.value
                      const email = (document.getElementById('register-email') as HTMLInputElement)?.value
                      const password = (document.getElementById('register-password') as HTMLInputElement)?.value
                      if (username && email && password) handleRegister(username, email, password)
                    }}
                    className="w-full bg-gradient-to-r from-green-500 to-emerald-600 hover:from-green-600 hover:to-emerald-700 py-3 rounded-xl font-bold"
                  >
                    ✨ Create Account
                  </Button>
                </TabsContent>
              </Tabs>
            </div>
          </DialogContent>
        </Dialog>

        {/* UDG Key Modal */}
        <Dialog open={showUdgModal} onOpenChange={setShowUdgModal}>
          <DialogContent className={`max-w-md ${
            isDarkMode ? 'bg-slate-900 border-slate-700' : 'bg-white border-slate-200'
          } rounded-2xl`}>
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2 text-xl font-bold">
                <Shield className="w-6 h-6 text-blue-400" />
                🔐 UDG Key Required
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-6">
              <div className={`p-4 rounded-xl ${
                isDarkMode ? 'bg-yellow-500/10 border border-yellow-500/20' : 'bg-yellow-100 border border-yellow-200'
              }`}>
                <p className={`font-medium ${isDarkMode ? 'text-yellow-300' : 'text-yellow-700'}`}>
                  ⚠️ Security Verification Required
                </p>
                <p className={`text-sm mt-1 ${isDarkMode ? 'text-yellow-400' : 'text-yellow-600'}`}>
                  Please enter your UDG Key to continue the upvote process at 85% completion.
                </p>
              </div>
              
              <div className="space-y-2">
                <label className={`text-sm font-bold ${isDarkMode ? 'text-slate-300' : 'text-slate-700'}`}>
                  🔑 UDG Key
                </label>
                <Input
                  value={udgKeyInput}
                  onChange={(e) => setUdgKeyInput(e.target.value)}
                  placeholder="UDG-XXXX-XXXX-XXXX"
                  className={`py-3 rounded-xl font-mono ${
                    isDarkMode 
                      ? 'bg-slate-800/50 border-slate-600 text-white' 
                      : 'bg-slate-50 border-slate-300 text-slate-900'
                  }`}
                />
                {udgError && (
                  <p className="text-red-400 text-sm font-medium">❌ {udgError}</p>
                )}
              </div>
              
              <div className={`p-3 rounded-xl ${
                isDarkMode ? 'bg-blue-500/10 border border-blue-500/20' : 'bg-blue-100 border border-blue-200'
              }`}>
                <p className={`text-xs ${isDarkMode ? 'text-blue-300' : 'text-blue-700'}`}>
                  💡 Production UDG Key: <code className="font-bold">UDG-PROD-2024-LIVE</code>
                </p>
              </div>
              
              <div className="flex gap-6">
                <Button
                  onClick={handleUdgKeySubmit}
                  className="flex-1 bg-gradient-to-r from-blue-500 to-purple-600 hover:from-blue-600 hover:to-purple-700 py-4 rounded-2xl font-bold text-base"
                >
                  🔓 Verify Key
                </Button>
                <Button
                  onClick={() => {
                    setShowUdgModal(false)
                    stopProcess()
                  }}
                  variant="outline"
                  className="flex-1 py-4 rounded-2xl font-bold text-base"
                >
                  ❌ Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </div>
      
      <Toaster />
    </div>
  )
}
